/**
 * WT Event Filter - JavaScript
 * Client-seitige Ereignistyp-Filterung für Webtrees Personenseiten
 * 
 * Version: 1.0.1
 */

(function() {
    'use strict';

    // Konfiguration: DOM-Selektoren für Kompatibilität mit verschiedenen Tabs/Themes
    const SELECTORS = {
        // Tab-Container (werden der Reihe nach versucht, erster Treffer gewinnt)
        tabRoot: [
            '#vesta_personal_facts_',
            '.tab-pane.active',
            '#individual-tabs',
            '#vesta-personal-facts',
            '#tab-facts',
            '.wt-tab-facts',
            '#content'
        ],
        // Individuelle Ereignis-/Fakten-Zeilen
        eventRow: [
            'tr:not(.collapse):has(.wt-fact-label)',
            'tr.wt-fact:not(.collapse)',
            '.wt-fact',
            'tr.wt-level-two-source',
            'tr.wt-level-two-note',
            'tr.wt-level-two-media'
        ],
        // Element, das das Ereignistyp-Label enthält
        typeLabel: [
            '.wt-fact-label',
            '.wt-fact-type',
            'th[scope="row"]',
            '.fact-type',
            '.label',
            '.tag'
        ]
    };

    // Debug-Modus
    const DEBUG = false;

    // Übersetzungen
    const TRANSLATIONS = {
        'de': {
            filterLabel: 'Ereignisse filtern:',
            clearAll: 'Alle löschen'
        },
        'en-GB': {
            filterLabel: 'Filter events:',
            clearAll: 'Clear all'
        },
        'en': {
            filterLabel: 'Filter events:',
            clearAll: 'Clear all'
        },
        'fr': {
            filterLabel: 'Filtrer les événements :',
            clearAll: 'Tout effacer'
        },
        'hu': {
            filterLabel: 'Események szűrése:',
            clearAll: 'Összes törlése'
        }
    };

    function getLanguage() {
        if (window.WT_EVENT_FILTER_LANG) {
            log('Using language from Webtrees:', window.WT_EVENT_FILTER_LANG);
            return window.WT_EVENT_FILTER_LANG;
        }
        
        const htmlLang = document.documentElement.getAttribute('lang');
        log('HTML lang attribute:', htmlLang);
        
        if (htmlLang && TRANSLATIONS[htmlLang]) {
            log('Using HTML lang:', htmlLang);
            return htmlLang;
        }
        
        if (htmlLang) {
            const langCode = htmlLang.split('-')[0];
            if (TRANSLATIONS[langCode]) {
                log('Using base language:', langCode);
                return langCode;
            }
        }
        
        log('Using fallback language: de');
        return 'de';
    }

    function translate(key) {
        const lang = getLanguage();
        const translations = TRANSLATIONS[lang] || TRANSLATIONS['de'];
        const translated = translations[key] || TRANSLATIONS['de'][key];
        log(`Translating "${key}" to "${translated}" (lang: ${lang})`);
        return translated;
    }

    function log(...args) {
        if (DEBUG) {
            console.log('[WT Event Filter]', ...args);
        }
    }

    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    function findElement(selectorArray, context = document) {
        for (const selector of selectorArray) {
            try {
                const element = context.querySelector(selector);
                if (element) {
                    log('Element gefunden mit Selektor:', selector);
                    return element;
                }
            } catch (e) {
                log('Ungültiger Selektor:', selector, e);
            }
        }
        return null;
    }

    function findElements(selectorArray, context = document) {
        for (const selector of selectorArray) {
            try {
                const elements = context.querySelectorAll(selector);
                if (elements.length > 0) {
                    log(elements.length, 'Elemente gefunden mit Selektor:', selector);
                    return Array.from(elements);
                }
            } catch (e) {
                log('Ungültiger Selektor:', selector, e);
            }
        }
        return [];
    }

    function getPersonXref() {
        const match = window.location.pathname.match(/\/individual\/([^\/]+)/);
        if (match) {
            log('XREF in URL gefunden:', match[1]);
            return match[1];
        }

        const xref = document.body.dataset.xref;
        if (xref) {
            log('XREF in Body-Data gefunden:', xref);
            return xref;
        }

        const wtPageTitle = document.querySelector('.wt-page-title');
        if (wtPageTitle) {
            const titleMatch = wtPageTitle.textContent.match(/\(([I0-9]+)\)/);
            if (titleMatch) {
                log('XREF im Seitentitel gefunden:', titleMatch[1]);
                return titleMatch[1];
            }
        }

        log('XREF konnte nicht gefunden werden');
        return null;
    }

    function getStorageKey() {
        const xref = getPersonXref();
        return xref ? `wtEventFilter:${xref}` : 'wtEventFilter:default';
    }

    function saveFilterState(selectedTypes) {
        try {
            const key = getStorageKey();
            localStorage.setItem(key, JSON.stringify(selectedTypes));
            log('Filterzustand gespeichert:', selectedTypes);
        } catch (e) {
            log('Konnte nicht in localStorage speichern:', e);
        }
    }

    function loadFilterState() {
        try {
            const key = getStorageKey();
            const stored = localStorage.getItem(key);
            const state = stored ? JSON.parse(stored) : [];
            log('Filterzustand geladen:', state);
            return state;
        } catch (e) {
            log('Konnte nicht aus localStorage laden:', e);
            return [];
        }
    }

    function init() {
        log('WT Event Filter wird initialisiert...');
        log('Aktuelle URL:', window.location.href);

        const delays = [0, 500, 1000, 2000];
        let attemptIndex = 0;

        function attemptInit() {
            const success = tryInit();
            
            if (!success && attemptIndex < delays.length) {
                const nextDelay = delays[attemptIndex];
                log(`Keine Ereignisse gefunden, erneuter Versuch in ${nextDelay}ms...`);
                attemptIndex++;
                setTimeout(attemptInit, nextDelay);
            }
        }

        attemptInit();
    }

    function tryInit() {
        const tabRoot = findElement(SELECTORS.tabRoot);
        if (!tabRoot) {
            log('Konnte Facts-Tab-Container nicht finden. Modul deaktiviert.');
            return false;
        }

        log('Tab-Root gefunden:', tabRoot);

        const eventRows = findElements(SELECTORS.eventRow, tabRoot);
        if (eventRows.length === 0) {
            log('Keine Ereignis-Zeilen gefunden. Versuchte Selektoren:', SELECTORS.eventRow);
            log('Tab-Root-Inhalt:', tabRoot.innerHTML.substring(0, 500));
            return false;
        }

        log(eventRows.length, 'Ereignis-Zeilen gefunden');

        const eventTypes = new Map();
        
        eventRows.forEach((row, index) => {
            if (row.tagName === 'TR') {
                const labelElement = row.querySelector('.wt-fact-label');
                if (!labelElement) {
                    log('Kein .wt-fact-label in Zeile', index);
                    return;
                }

                const rawLabel = labelElement.textContent.trim();
                if (!rawLabel) {
                    log('Leeres Label in Zeile', index);
                    return;
                }

                log('Ereignistyp gefunden:', rawLabel, 'in Zeile', index);

                const normalizedType = rawLabel.toLowerCase().replace(/\s+/g, ' ');
                
                if (!eventTypes.has(normalizedType)) {
                    eventTypes.set(normalizedType, {
                        label: rawLabel,
                        rows: []
                    });
                }
                eventTypes.get(normalizedType).rows.push(row);
            } 
            else if (row.classList && row.classList.contains('wt-fact')) {
                const labelElement = row.querySelector('.wt-fact-label');
                if (!labelElement) {
                    log('Kein Label-Element in Fakten-Div', index);
                    return;
                }

                const rawLabel = labelElement.textContent.trim();
                if (!rawLabel) {
                    log('Leeres Label in Fakten-Div', index);
                    return;
                }

                log('Ereignistyp gefunden:', rawLabel, 'in Fakten-Div', index);

                const normalizedType = rawLabel.toLowerCase().replace(/\s+/g, ' ');
                
                if (!eventTypes.has(normalizedType)) {
                    eventTypes.set(normalizedType, {
                        label: rawLabel,
                        rows: []
                    });
                }
                eventTypes.get(normalizedType).rows.push(row);
            }
        });

        if (eventTypes.size === 0) {
            log('Keine Ereignistypen nach Filterung erkannt');
            return false;
        }

        log('Ereignistypen erkannt:', Array.from(eventTypes.keys()));

        const filterContainer = createFilterUI(eventTypes);
        
        const firstEvent = eventRows[0];
        const eventTable = firstEvent.closest('table');
        
        if (eventTable) {
            eventTable.parentNode.insertBefore(filterContainer, eventTable);
            log('Filter-UI vor Tabelle eingefügt');
        } else {
            tabRoot.insertBefore(filterContainer, tabRoot.firstChild);
            log('Filter-UI bei Tab-Root eingefügt');
        }

        const savedState = loadFilterState();
        if (savedState.length > 0) {
            savedState.forEach(type => {
                const checkbox = filterContainer.querySelector(`input[data-type="${type}"]`);
                if (checkbox) {
                    checkbox.checked = true;
                    log('Checkbox wiederhergestellt:', type);
                }
            });
            applyFilter(eventTypes, filterContainer);
        }

        log('Initialisierung abgeschlossen!');
        return true;
    }

    function createFilterUI(eventTypes) {
        const container = document.createElement('div');
        container.className = 'wt-event-filter';
        
        const label = document.createElement('span');
        label.className = 'wt-event-filter-label';
        label.textContent = translate('filterLabel');
        container.appendChild(label);

        const checkboxContainer = document.createElement('div');
        checkboxContainer.className = 'wt-event-filter-checkboxes';

        const sortedTypes = Array.from(eventTypes.entries())
            .sort((a, b) => a[1].label.localeCompare(b[1].label));

        sortedTypes.forEach(([normalizedType, data]) => {
            const wrapper = document.createElement('label');
            wrapper.className = 'wt-event-filter-checkbox';

            const checkbox = document.createElement('input');
            checkbox.type = 'checkbox';
            checkbox.dataset.type = normalizedType;
            checkbox.setAttribute('aria-label', `Filter ${data.label}`);

            const labelSpan = document.createElement('span');
            labelSpan.textContent = data.label;

            wrapper.appendChild(checkbox);
            wrapper.appendChild(labelSpan);
            checkboxContainer.appendChild(wrapper);

            checkbox.addEventListener('change', debounce(() => {
                applyFilter(eventTypes, container);
            }, 75));
        });

        container.appendChild(checkboxContainer);

        const clearButton = document.createElement('button');
        clearButton.type = 'button';
        clearButton.className = 'wt-event-filter-clear';
        clearButton.textContent = translate('clearAll');
        clearButton.setAttribute('aria-label', translate('clearAll'));
        clearButton.addEventListener('click', () => {
            clearAllFilters(container, eventTypes);
        });
        container.appendChild(clearButton);

        return container;
    }

    function applyFilter(eventTypes, container) {
        const checkboxes = container.querySelectorAll('input[type="checkbox"]');
        const selectedTypes = Array.from(checkboxes)
            .filter(cb => cb.checked)
            .map(cb => cb.dataset.type);

        saveFilterState(selectedTypes);

        if (selectedTypes.length === 0) {
            eventTypes.forEach(data => {
                data.rows.forEach(row => row.classList.remove('is-hidden'));
            });
            log('Alle Ereignisse anzeigen (kein Filter)');
            return;
        }

        eventTypes.forEach((data, type) => {
            const shouldShow = selectedTypes.includes(type);
            data.rows.forEach(row => {
                if (shouldShow) {
                    row.classList.remove('is-hidden');
                } else {
                    row.classList.add('is-hidden');
                }
            });
        });

        log('Filter angewendet:', selectedTypes);
    }

    function clearAllFilters(container, eventTypes) {
        log('Alle Filter werden gelöscht');
        const checkboxes = container.querySelectorAll('input[type="checkbox"]');
        checkboxes.forEach(cb => cb.checked = false);
        applyFilter(eventTypes, container);
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();