<?php

/**
 * webtrees: online genealogy
 * Copyright (C) 2024 webtrees development team
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

declare(strict_types=1);

namespace WtEventFilter;

use Fisharebest\Webtrees\I18N;
use Fisharebest\Webtrees\Module\AbstractModule;
use Fisharebest\Webtrees\Module\ModuleCustomInterface;
use Fisharebest\Webtrees\Module\ModuleCustomTrait;
use Fisharebest\Webtrees\Module\ModuleGlobalInterface;
use Fisharebest\Webtrees\Module\ModuleGlobalTrait;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Server\RequestHandlerInterface;

use function response;

/**
 * WT Event Filter Modul
 * 
 * Fügt Checkbox-Filter für Ereignistypen auf Personenseiten hinzu.
 * Kompatibel mit Standard "Tatsachen und Ereignisse" Tab und Vesta Facts and events Modul.
 */
class WtEventFilterModule extends AbstractModule implements 
    ModuleCustomInterface, 
    ModuleGlobalInterface,
    RequestHandlerInterface
{
    use ModuleCustomTrait;
    use ModuleGlobalTrait;

    /**
     * Modul-Konstanten
     */
    private const MODULE_NAME = 'WT Event Filter';
    private const MODULE_VERSION = '2.2.4.0.0';
    
    /**
     * Übersetzungs-Keys als Konstanten
     */
    private const TRANS_DESCRIPTION = 'Adds checkbox filters for event types on individual pages. Compatible with standard tabs and Vesta Facts and events.';
    private const TRANS_AUTHOR = 'Thomas Schiller (with the help of AI Sonnet)';
    
    /**
     * Bootstrap-Methode - wird beim Laden des Moduls aufgerufen
     *
     * @return void
     */
    public function boot(): void
    {
        // Aktuell keine Views benötigt, aber vorbereitet für zukünftige Erweiterungen
        // View::registerNamespace($this->name(), $this->resourcesFolder() . 'views/');
    }
    
    /**
     * Wie soll dieses Modul im Kontrollfeld usw. bezeichnet werden?
     *
     * @return string
     */
    public function title(): string
    {
        return self::MODULE_NAME;
    }

    /**
     * Eine Beschreibung dessen, was dieses Modul tut.
     *
     * @return string
     */
    public function description(): string
    {
        return I18N::translate(self::TRANS_DESCRIPTION);
    }

    /**
     * Die Person oder Organisation, die dieses Modul erstellt hat.
     *
     * @return string
     */
    public function customModuleAuthorName(): string
    {
        return I18N::translate(self::TRANS_AUTHOR);
    }

    /**
     * Die Version dieses Moduls.
     *
     * @return string
     */
    public function customModuleVersion(): string
    {
        return self::MODULE_VERSION;
    }

    /**
     * Eine URL, die die neueste Version dieses Moduls bereitstellt.
     *
     * @return string
     */
    public function customModuleLatestVersionUrl(): string
    {
        return 'https://wt-module.schitho.at';
    }

    /**
     * Wo man Unterstützung für dieses Modul erhält. Vielleicht ein GitHub-Repository?
     *
     * @return string
     */
    public function customModuleSupportUrl(): string
    {
        return 'https://wt-module.schitho.at';
    }

    /**
     * Zusätzliche/aktualisierte Übersetzungen.
     *
     * @param string $language
     *
     * @return array<string,string>
     */
    public function customTranslations(string $language): array
    {
        $file = $this->resourcesFolder() . 'lang/' . $language . '.php';
        
        return file_exists($file) ? require $file : [];
    }

    /**
     * Roher Inhalt, der zum Head-Bereich jeder Seite hinzugefügt wird.
     * Typischerweise werden dies CSS und JS sein.
     *
     * @return string
     */
    public function headContent(): string
    {
        $css_url = $this->assetUrl('css/filter.css');
        $js_url = $this->assetUrl('js/filter.js');
        
        // Aktuelle Sprache direkt an JavaScript übergeben
        $lang = I18N::languageTag();
        
        return 
            '<link rel="stylesheet" href="' . e($css_url) . '">' . PHP_EOL .
            '<script>window.WT_EVENT_FILTER_LANG = ' . json_encode($lang) . ';</script>' . PHP_EOL .
            '<script src="' . e($js_url) . '" defer></script>';
    }

    /**
     * Wo sind unsere Ressourcen?
     *
     * @return string
     */
    public function resourcesFolder(): string
    {
        return __DIR__ . '/resources/';
    }

    /**
     * URL für ein Asset abrufen
     *
     * @param string $asset
     *
     * @return string
     */
    public function assetUrl(string $asset): string
    {
        // Webtrees Asset-URL-Helper für korrekte Modul-Asset-URLs verwenden
        return route('module', [
            'module' => $this->name(),
            'action' => 'Asset',
            'asset'  => $asset,
        ]);
    }

    /**
     * Asset-Anfragen behandeln
     *
     * @param ServerRequestInterface $request
     *
     * @return ResponseInterface
     */
    public function handle(ServerRequestInterface $request): ResponseInterface
    {
        $asset = $request->getQueryParams()['asset'] ?? '';
        
        // Sicherheit: nur bestimmte Asset-Pfade erlauben
        if (!preg_match('/^(css|js)\/[a-z0-9._-]+\.(css|js)$/i', $asset)) {
            return response('Not found', 404);
        }

        $file = $this->resourcesFolder() . $asset;
        
        if (!file_exists($file)) {
            return response('Not found', 404);
        }

        $content = file_get_contents($file);
        $extension = pathinfo($file, PATHINFO_EXTENSION);
        
        $mime_types = [
            'css' => 'text/css',
            'js'  => 'application/javascript',
        ];
        
        $mime_type = $mime_types[$extension] ?? 'text/plain';
        
        return response($content)
            ->withHeader('Content-Type', $mime_type)
            ->withHeader('Cache-Control', 'public, max-age=31536000');
    }
}

return new WtEventFilterModule();
