<?php

declare(strict_types=1);

namespace MyVendor\Webtrees\Module\BurialPlacesReport;


use function view;               // GANZ WICHTIG fuer den View-Helper im Namespace


use Fisharebest\Webtrees\Auth;
use Fisharebest\Webtrees\FlashMessages;
use Fisharebest\Webtrees\Http\RequestHandlers\ControlPanel;
use Fisharebest\Webtrees\Http\RequestHandlers\ModulesAllPage;
use Fisharebest\Webtrees\I18N;
use Fisharebest\Webtrees\Module\AbstractModule;
use Fisharebest\Webtrees\Module\ModuleConfigInterface;
use Fisharebest\Webtrees\Module\ModuleConfigTrait;
use Fisharebest\Webtrees\Module\ModuleCustomInterface;
use Fisharebest\Webtrees\Module\ModuleCustomTrait;
use Fisharebest\Webtrees\Module\ModuleGlobalInterface;
use Fisharebest\Webtrees\Module\ModuleGlobalTrait;
use Fisharebest\Webtrees\Module\ModuleListInterface;
use Fisharebest\Webtrees\Module\ModuleListTrait;
use Fisharebest\Webtrees\Registry;
use Fisharebest\Webtrees\Tree;
use Fisharebest\Webtrees\Validator;
use Fisharebest\Webtrees\View;
use Illuminate\Database\Capsule\Manager as DB;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

require_once __DIR__ . '/module_helper.php';

class BurialPlacesReportModule extends AbstractModule implements 
    ModuleCustomInterface,
    ModuleListInterface,
    ModuleConfigInterface,
    ModuleGlobalInterface
{
    use ModuleCustomTrait;
    use ModuleListTrait;
    use ModuleConfigTrait;
    use ModuleGlobalTrait;

    public const CUSTOM_VERSION = '2.2.4.1.2';
    public const CUSTOM_WEBSITE = 'https://wt-module.schitho.at';

    // Standard-Einstellungen
    private const DEFAULT_FALLBACK_ORDER = 'AGNC,PLAC,NOTE';
    private const DEFAULT_PLAC_KEYWORDS = 'Friedhof,Cemetery,CimetiÃ¨re,TemetÅ‘';
    private const DEFAULT_DIAGNOSIS_MODE = '0';
    private const DEFAULT_SHOW_AGNC_ALWAYS = '0';
    
    // Validierungskonstanten
    private const MAX_KEYWORDS = 20;
    private const MAX_KEYWORD_LENGTH = 50;
    private const MAX_TOTAL_LENGTH = 250;  // Sicherheitspuffer fuer VARCHAR(255)

    public function title(): string
    {
        return I18N::translate('Burial Places Report');
    }

    public function description(): string
    {
        return I18N::translate('A list of burials, grouped by burial sites and institutions, with configurable data sources and filters.');
    }
		
	public function getStylesheets(): array
	{
		return [$this->assetUrl('css/menu-icons.css')];
	}


	
	public function customModuleAuthorName(): string
    {
        return I18N::translate('Thomas Schiller (with the help of the AI tools Sonnet and ChatGPT)');
    }

    public function customModuleSupportUrl(): string
    {
        return self::CUSTOM_WEBSITE;
    }

    public function customModuleLatestVersionUrl(): string
    {
        return self::CUSTOM_WEBSITE;
    }
	
    public function customModuleVersion(): string
    {
        return self::CUSTOM_VERSION;
    }

    public function boot(): void
    {
        View::registerNamespace($this->name(), $this->resourcesFolder() . 'views/');
        
        Registry::routeFactory()->routeMap()
            ->get('burial-places-list', '/tree/{tree}/burial-places', $this)
            ->allows('POST');
        
        // CSV-Export Route
        Registry::routeFactory()->routeMap()
            ->post('burial-places-csv', '/tree/{tree}/burial-places/export', [$this, 'exportCsvAction']);

    }

    public function resourcesFolder(): string
    {
        return __DIR__ . '/resources/';
    }

    public function customTranslations(string $language): array
    {
        $file = $this->resourcesFolder() . 'lang/' . $language . '.php';
        
        return file_exists($file) ? require $file : [];
    }

    public function listMenuClass(): string
    {
        return 'menu-list-burial';
    }

    public function listUrlAttributes(): array
    {
        return [];
    }

    public function listUrl(Tree $tree, array $parameters = []): string
    {
        return route('burial-places-list', ['tree' => $tree->name()] + $parameters);
    }
	
	public function headContent(): string
{
    // 1) Bestehender Inhalt (z. B. AGNC-Block)
    $html = view($this->name() . '::agnc-always-show', [
        'module_name' => $this->name(),
    ]);

    // 2) Pfade vorbereiten
    $css_dir_fs  = rtrim($this->resourcesFolder(), '/\\') . '/css/';
    $module_dir  = basename(dirname(__FILE__)); // Modulordner
    
    // Base-Pfad ermitteln (universelle Lösung für alle Installationsszenarien)
    $scriptName = $_SERVER['SCRIPT_NAME'] ?? '/index.php';
    $basePath = dirname($scriptName);
    
    // Normalisierung für Root-Pfad (/, \, .) werden zu '' (leer)
    if ($basePath === '/' || $basePath === '\\' || $basePath === '.') {
        $basePath = '';
    }
    
    // Icons-URL mit Base-Path (funktioniert bei Root, Subdomain UND Unterverzeichnis)
    $icons_base = $basePath . '/modules_v4/' . $module_dir . '/resources/icons/';

    // 3) AKTUELLES THEME ERMITTELN und nur passende CSS laden
    $current_theme = '';
    try {
        // Versuche das Theme Ã¼ber den Request zu ermitteln
        $request = Registry::container()->get(ServerRequestInterface::class);
        $theme_attr = $request->getAttribute('theme');
        
        if ($theme_attr !== null && method_exists($theme_attr, 'name')) {
            $current_theme = $theme_attr->name();
        }
    } catch (\Exception $e) {
        // Fallback: Theme konnte nicht ermittelt werden
        $current_theme = '';
    }

    // 4) NUR die CSS-Datei des aktiven Themes laden
    $files = [];
    $theme_css_found = false;

    // Wenn Theme ermittelt werden konnte: nur dessen CSS laden
    if ($current_theme !== '') {
        $theme_css = 'menu-icons.' . $current_theme . '.css';
        if (is_file($css_dir_fs . $theme_css)) {
            $files[] = $theme_css;
            $theme_css_found = true;
        }
    }

    // FALLBACK 1: Wenn kein Theme-spezifisches CSS existiert, versuche universal.css
    if (!$theme_css_found && is_file($css_dir_fs . 'menu-icons.universal.css')) {
        $files[] = 'menu-icons.universal.css';
        $theme_css_found = true;
    }

    // FALLBACK 2: Wenn gar nichts gefunden wurde, lade alle verfÃ¼gbaren Theme-CSS
    if (!$theme_css_found) {
        $all = glob($css_dir_fs . 'menu-icons.*.css');
        if (is_array($all)) {
            foreach ($all as $abs) {
                $bn = basename($abs);
                
                if ($bn === 'menu-icons.universal.css' || 
                    $bn === 'menu-icons.none.css' || 
                    str_starts_with($bn, 'menu-icons.template-')) {
                    continue;
                }
                
                if (preg_match('/^menu-icons\..+\.css$/', $bn)) {
                    $files[] = $bn;
                }
            }
        }
    }

    // 5) Inhalte zusammenbauen + Icon-Pfade korrigieren
    $bundle = '';
    foreach ($files as $file) {
        $path = $css_dir_fs . $file;
        if (!is_file($path)) {
            continue;
        }

        $css = file_get_contents($path);
        if ($css === false) {
            continue;
        }

        // Pfadfix: relative -> absolute (fuer inline <style>)
        // a) ../icons/webtrees/... --> /modules_v4/<modul>/resources/icons/webtrees/...
        $css = str_replace('../icons/webtrees/', $icons_base . 'webtrees/', $css);
        // b) ../icons/... --> /modules_v4/<modul>/resources/icons/...
        $css = str_replace('../icons/', $icons_base, $css);
        // c) Backslashes normalisieren (Windows)
        $css = str_replace('\\', '/', $css);

        $bundle .= "\n/* {$file} (Active Theme: {$current_theme}) */\n" . $css . "\n";
    }

    // 6) Inline einfÃ¼gen mit Debug-Info
    if ($bundle !== '') {
        $html .= "\n<!-- Burial Places Module - Icon CSS -->\n";
        $html .= "<!-- Detected Theme: " . ($current_theme ?: 'unknown') . " -->\n";
        $html .= "<!-- Loaded CSS files: " . implode(', ', $files) . " -->\n";
        $html .= "<style>\n" . $bundle . "\n</style>\n";
    }

    return $html;
}



    /**
     * Admin-Action fuer GET-Request (Konfigurationsseite anzeigen)
     */
    public function getAdminAction(ServerRequestInterface $request): ResponseInterface
    {
        $this->layout = 'layouts/administration';

        $tree = Validator::attributes($request)->treeOptional();
        
        // Wenn kein Baum ausgewaehlt, zeige Baum-Auswahl
        if ($tree === null) {
            return $this->viewResponse($this->name() . '::admin-tree-select', [
                'title' => $this->title(),
                'breadcrumbs' => $this->getBreadcrumbs(),
                'module_name' => $this->name(),
            ]);
        }

        // Baum-spezifische Einstellungen laden
        $fallback_order = $this->getTreePreference($tree, 'fallback_order', self::DEFAULT_FALLBACK_ORDER);
        $plac_keywords = $this->getTreePreference($tree, 'plac_keywords', self::DEFAULT_PLAC_KEYWORDS);
        $diagnosis_mode = $this->getTreePreference($tree, 'diagnosis_mode', self::DEFAULT_DIAGNOSIS_MODE);
        $show_agnc_always = $this->getTreePreference($tree, 'show_agnc_always', self::DEFAULT_SHOW_AGNC_ALWAYS);

        return $this->viewResponse($this->name() . '::admin', [
            'title' => $this->title(),
            'tree' => $tree,
            'module_name' => $this->name(),
            'breadcrumbs' => $this->getBreadcrumbs(),
            'fallback_order' => $fallback_order,
            'plac_keywords' => $plac_keywords,
            'diagnosis_mode' => $diagnosis_mode,
            'show_agnc_always' => $show_agnc_always,
        ]);
    }

    /**
     * Admin-Action fuer POST-Request (Einstellungen speichern)
     */
    public function postAdminAction(ServerRequestInterface $request): ResponseInterface
    {
        $tree = Validator::attributes($request)->tree();
        $params = (array) $request->getParsedBody();

        // Die Reihenfolge kommt aus dem hidden field
        $fallback_order = $params['fallback_order'] ?? self::DEFAULT_FALLBACK_ORDER;
        $plac_keywords_raw = $params['plac_keywords'] ?? self::DEFAULT_PLAC_KEYWORDS;
        $diagnosis_mode = ($params['diagnosis_mode'] ?? '0') === '1' ? '1' : '0';
        $show_agnc_always = ($params['show_agnc_always'] ?? '0') === '1' ? '1' : '0';

        // Validierung der Fallback-Reihenfolge
        $valid_sources = ['AGNC', 'NOTE', 'PLAC'];
        $sources = array_map('trim', explode(',', strtoupper($fallback_order)));
        $sources = array_filter($sources, fn($s) => in_array($s, $valid_sources));
        $fallback_order = implode(',', array_unique($sources));

        if (empty($fallback_order)) {
            $fallback_order = self::DEFAULT_FALLBACK_ORDER;
        }

        // Validierung der Keywords
        $validation_result = $this->validateKeywords($plac_keywords_raw);
        
        if (!$validation_result['valid']) {
            FlashMessages::addMessage($validation_result['error'], 'danger');
            
            return redirect(route('module', [
                'module' => $this->name(),
                'action' => 'Admin',
                'tree' => $tree->name()
            ]));
        }
        
        $plac_keywords = $validation_result['keywords'];

        // Einstellungen speichern
        $this->setTreePreference($tree, 'fallback_order', $fallback_order);
        $this->setTreePreference($tree, 'plac_keywords', $plac_keywords);
        $this->setTreePreference($tree, 'diagnosis_mode', $diagnosis_mode);
        $this->setTreePreference($tree, 'show_agnc_always', $show_agnc_always);

        FlashMessages::addMessage(
            I18N::translate('The preferences for the module "%s" have been updated.', $this->title()),
            'success'
        );

        return redirect(route('module', [
            'module' => $this->name(),
            'action' => 'Admin',
            'tree' => $tree->name()
        ]));
    }

    /**
     * Validiert und bereinigt die Keyword-Eingabe
     */
    private function validateKeywords(string $input): array
    {
        // Entferne Whitespace und teile bei Komma, Semikolon oder Zeilenumbruch
        $keywords = preg_split('/[,;\n\r]+/', $input);
        $keywords = array_map('trim', $keywords);
        $keywords = array_filter($keywords, fn($k) => $k !== '');
        
        // Entferne Duplikate (case-insensitive)
        $unique_keywords = [];
        $seen = [];
        foreach ($keywords as $keyword) {
            $lower = mb_strtolower($keyword);
            if (!isset($seen[$lower])) {
                $unique_keywords[] = $keyword;
                $seen[$lower] = true;
            }
        }
        $keywords = $unique_keywords;
        
        // Pruefe Anzahl der Keywords
        if (count($keywords) > self::MAX_KEYWORDS) {
            return [
                'valid' => false,
                'error' => I18N::translate('Too many keywords. Maximum allowed: %s', I18N::number(self::MAX_KEYWORDS)),
                'keywords' => ''
            ];
        }
        
        // Pruefe Laenge einzelner Keywords
        foreach ($keywords as $keyword) {
            if (mb_strlen($keyword) > self::MAX_KEYWORD_LENGTH) {
                return [
                    'valid' => false,
                    'error' => I18N::translate('Keyword "%s" is too long. Maximum length: %s characters', 
                        mb_substr($keyword, 0, 20) . '...', 
                        I18N::number(self::MAX_KEYWORD_LENGTH)),
                    'keywords' => ''
                ];
            }
        }
        
        // Erstelle bereinigte Keyword-Liste
        $cleaned = implode(',', $keywords);
        
        // Pruefe Gesamtlaenge
        if (mb_strlen($cleaned) > self::MAX_TOTAL_LENGTH) {
            return [
                'valid' => false,
                'error' => I18N::translate('Total length of keywords is too long. Maximum: %s characters. Current: %s characters', 
                    I18N::number(self::MAX_TOTAL_LENGTH),
                    I18N::number(mb_strlen($cleaned))),
                'keywords' => ''
            ];
        }
        
        return [
            'valid' => true,
            'error' => '',
            'keywords' => $cleaned
        ];
    }

	/**
     * Reset-Action: Standardwerte wiederherstellen
     */
    public function getResetDefaultsAction(ServerRequestInterface $request): ResponseInterface
    {
        $tree = Validator::attributes($request)->tree();
        
        // Auf Standardwerte zuruecksetzen
        $this->setTreePreference($tree, 'fallback_order', self::DEFAULT_FALLBACK_ORDER);
        $this->setTreePreference($tree, 'plac_keywords', self::DEFAULT_PLAC_KEYWORDS);
        $this->setTreePreference($tree, 'diagnosis_mode', self::DEFAULT_DIAGNOSIS_MODE);
        $this->setTreePreference($tree, 'show_agnc_always', self::DEFAULT_SHOW_AGNC_ALWAYS);
        
        FlashMessages::addMessage(
            I18N::translate('Settings have been reset to default values.'),
            'success'
        );
        
        return redirect(route('module', [
            'module' => $this->name(),
            'action' => 'Admin',
            'tree' => $tree->name()
        ]));
    }

    /**
     * Breadcrumb-Navigation
     */
    private function getBreadcrumbs(): array
    {
        return [
            route(ControlPanel::class) => I18N::translate('Control panel'),
            route(ModulesAllPage::class) => I18N::translate('Modules'),
            $this->title() => $this->title(),
        ];
    }

    /**
     * Baum-spezifische Preference laden
     */
    private function getTreePreference(Tree $tree, string $key, string $default): string
    {
        $setting_name = 'burial_places_' . $key;
        return $tree->getPreference($setting_name, $default);
    }

    /**
     * Baum-spezifische Preference speichern
     */
    private function setTreePreference(Tree $tree, string $key, string $value): void
    {
        $setting_name = 'burial_places_' . $key;
        $tree->setPreference($setting_name, $value);
    }

    public function handle(ServerRequestInterface $request): ResponseInterface
    {
        $tree = Validator::attributes($request)->tree();
        
        if (!Auth::isManager($tree)) {
            Auth::checkComponentAccess($this, ModuleListInterface::class, $tree, $request->getAttribute('user'));
        }

        $method = $request->getMethod();
        
        if ($method === 'POST') {
            $params = (array) $request->getParsedBody();
            
            if (($params['action'] ?? '') === 'csv') {
                return $this->exportCsvAction($request);
            }
            
            return $this->postListAction($request);
        }
        
        return $this->getListAction($request);
    }

    private function getListAction(ServerRequestInterface $request): ResponseInterface
    {
        $tree = Validator::attributes($request)->tree();
        
        $this->layout = 'layouts/default';

        // Einstellungen laden
        $diagnosis_mode = $this->getTreePreference($tree, 'diagnosis_mode', self::DEFAULT_DIAGNOSIS_MODE) === '1';

        return $this->viewResponse($this->name() . '::list-page', [
            'title' => $this->title(),
            'tree'  => $tree,
            'module' => $this->name(),
            'institutions' => [],
            'place_filter' => '',
            'institution_filter' => '',
            'sort_by' => 'place',
            'burial_place_filter' => 'with_info',
            'diagnosis_mode' => $diagnosis_mode,
            'report_generated' => false,
        ]);
    }

    private function postListAction(ServerRequestInterface $request): ResponseInterface
    {
        $tree = Validator::attributes($request)->tree();
        $params = (array) $request->getParsedBody();
        
        $place_filter = $params['place_filter'] ?? '';
        $institution_filter = $params['institution_filter'] ?? '';
        $sort_by = $params['sort_by'] ?? 'place';
        
        // Neuer Dropdown-Filter mit Backward-Kompatibilitaet
        $burial_place_filter = $params['burial_place_filter'] ?? 'with_info';
        
        // Backward-Kompatibilitaet: Alte Checkbox-Parameter auf neues System mappen
        if (isset($params['hide_none_filter'])) {
            $burial_place_filter = ($params['hide_none_filter'] === '1') ? 'with_info' : 'all';
        }

        $institutions = $this->collectInstitutions($tree, $place_filter, $institution_filter, $sort_by, $burial_place_filter);

        $this->layout = 'layouts/default';

        // Einstellungen laden
        $diagnosis_mode = $this->getTreePreference($tree, 'diagnosis_mode', self::DEFAULT_DIAGNOSIS_MODE) === '1';

        return $this->viewResponse($this->name() . '::list-page', [
            'title' => $this->title(),
            'tree'  => $tree,
            'module' => $this->name(),
            'institutions' => $institutions,
            'place_filter' => $place_filter,
            'institution_filter' => $institution_filter,
            'sort_by' => $sort_by,
            'burial_place_filter' => $burial_place_filter,
            'diagnosis_mode' => $diagnosis_mode,
            'report_generated' => true,
        ]);
    }

    /**
     * Sammelt alle Bestattungsorte mit konfigurierbarer Fallback-Logik
     */
    private function collectInstitutions(Tree $tree, string $place_filter, string $institution_filter, string $sort_by, string $burial_place_filter = 'with_info'): array
    {
        $institutions = [];

        // Einstellungen laden
        $fallback_order = $this->getTreePreference($tree, 'fallback_order', self::DEFAULT_FALLBACK_ORDER);
        $plac_keywords = $this->getTreePreference($tree, 'plac_keywords', self::DEFAULT_PLAC_KEYWORDS);
        $diagnosis_mode = $this->getTreePreference($tree, 'diagnosis_mode', self::DEFAULT_DIAGNOSIS_MODE) === '1';

        // Keyword-Liste vorbereiten
        $keywords = array_map('trim', explode(',', $plac_keywords));
        $keywords = array_filter($keywords);

        // Fallback-Reihenfolge vorbereiten
        $sources = array_map('trim', explode(',', strtoupper($fallback_order)));

        $query = DB::table('individuals')
            ->where('i_file', '=', $tree->id())
            ->select(['i_id', 'i_gedcom']);

        foreach ($query->get() as $row) {
            $individual = Registry::individualFactory()->make($row->i_id, $tree, $row->i_gedcom);
            
            if ($individual === null || !$individual->canShow()) {
                continue;
            }
            
            foreach ($individual->facts(['BURI']) as $burial) {
                $plac = $burial->place()->gedcomName();
                
                // Platzfilter anwenden
                if ($place_filter !== '' && stripos($plac, $place_filter) === false) {
                    continue;
                }

                // Institution ueber Fallback-Chain ermitteln
                $result = $this->getInstitutionFromBurial($burial, $sources, $keywords, $tree);
                $institution = $result['value'];
                $source_tag = $result['source'];
                
                // Neuer Dropdown-Filter anwenden
                if ($burial_place_filter === 'with_info' && $source_tag === 'NONE') {
                    // Nur mit Friedhofsinformationen: NONE ausblenden
                    continue;
                } elseif ($burial_place_filter === 'without_info' && $source_tag !== 'NONE') {
                    // Nur ohne Friedhofsinformationen: Nur NONE anzeigen
                    continue;
                }
                // Bei 'all' wird nichts gefiltert
                
                // Institutionsfilter anwenden
                if ($institution_filter !== '' && stripos($institution, $institution_filter) === false) {
                    continue;
                }

                $normalized_institution = $this->normalizeInstitution($institution);
                
                // Schluessel fuer Display (Institution + Ort)
                $display_key = $normalized_institution;
                if ($plac !== '') {
                    $display_key .= ' (' . $plac . ')';
                }
                
                // Normalisierter Schluessel fuer Gruppierung
                $group_key = $normalized_institution . '|' . $plac;
                
                if (!isset($institutions[$group_key])) {
                    $institutions[$group_key] = [
                        'institution' => $institution,
                        'normalized' => $normalized_institution,
                        'place' => $plac,
                        'display_key' => $display_key,
                        'individuals' => [],
                    ];
                }
                
                $institutions[$group_key]['individuals'][] = [
                    'individual' => $individual,
                    'burial' => $burial,
                    'source_tag' => $diagnosis_mode ? $source_tag : '',
                ];
            }
        }

        // Nach Institution oder Ort sortieren
        if ($sort_by === 'institution') {
            uasort($institutions, function ($a, $b) {
                $inst_cmp = strcasecmp($a['normalized'], $b['normalized']);
                if ($inst_cmp !== 0) {
                    return $inst_cmp;
                }
                return strcasecmp($a['place'], $b['place']);
            });
        } else {
            uasort($institutions, function ($a, $b) {
                $place_cmp = strcasecmp($a['place'], $b['place']);
                if ($place_cmp !== 0) {
                    return $place_cmp;
                }
                return strcasecmp($a['normalized'], $b['normalized']);
            });
        }

        return $institutions;
    }

    private function normalizeInstitution(string $institution): string
    {
        $normalized = trim($institution);
        $normalized = preg_replace('/\s+/', ' ', $normalized);
        return $normalized;
    }

    /**
     * Ermittelt die Institution aus einem Begraebnis-Fakt ueber die konfigurierte Fallback-Chain
     * 
     * @param object $burial Der Begraebnis-Fakt
     * @param array $sources Die Fallback-Reihenfolge der Quellen (z.B. ['AGNC', 'PLAC', 'NOTE'])
     * @param array $keywords Die Keywords fuer PLAC-Filterung
     * @param Tree $tree Der aktuelle Stammbaum
     * @return array Array mit ['value' => string, 'source' => string]
     */
    private function getInstitutionFromBurial(object $burial, array $sources, array $keywords, Tree $tree): array
    {
        $gedcom = $burial->gedcom();
        
        foreach ($sources as $source) {
            switch ($source) {
                case 'AGNC':
                    // AGNC direkt aus dem GEDCOM extrahieren
                    if (preg_match('/\n2 AGNC (.+)/', $gedcom, $match)) {
                        $agnc = trim($match[1]);
                        if ($agnc !== '') {
                            return ['value' => $agnc, 'source' => 'AGNC'];
                        }
                    }
                    break;
                    
                case 'PLAC':
                    // PLAC-Wert auf Keywords pruefen
                    $plac = $burial->place()->gedcomName();
                    if ($plac !== '' && !empty($keywords)) {
                        foreach ($keywords as $keyword) {
                            if (stripos($plac, $keyword) !== false) {
                                return ['value' => $plac, 'source' => 'PLAC'];
                            }
                        }
                    }
                    break;
                    
                case 'NOTE':
                    // NOTE-Eintraege durchsuchen
                    if (preg_match_all('/\n2 NOTE (.+)/m', $gedcom, $matches, PREG_SET_ORDER)) {
                        foreach ($matches as $match) {
                            $noteValue = trim($match[1]);
                            
                            // Pruefen ob es eine Referenz ist (@XREF@)
                            if (preg_match('/@(.+)@/', $noteValue, $refMatch)) {
                                $xref = $refMatch[1];
                                $noteRecord = Registry::noteFactory()->make($xref, $tree);
                                
                                if ($noteRecord !== null) {
                                    $noteText = trim($noteRecord->getNote());
                                    if ($noteText !== '') {
                                        // Erste Zeile der Notiz extrahieren
                                        $firstLine = strtok($noteText, "\n");
                                        $firstLine = trim($firstLine);
                                        
                                        // Pruefen ob erste Zeile ein Friedhofs-Schluesselwort enthaelt
                                        if (!empty($keywords)) {
                                            foreach ($keywords as $keyword) {
                                                if (stripos($firstLine, $keyword) !== false) {
                                                    return ['value' => $firstLine, 'source' => 'NOTE'];
                                                }
                                            }
                                        }
                                    }
                                }
                            } else {
                                // Inline-Notiz - mit CONT/CONC zusammenbauen
                                $fullNote = $noteValue;
                                
                                $notePos = strpos($gedcom, "\n2 NOTE " . $noteValue);
                                if ($notePos !== false) {
                                    $remainingGedcom = substr($gedcom, $notePos + strlen("\n2 NOTE " . $noteValue));
                                    
                                    if (preg_match_all('/\n3 (CONT|CONC) (.*)/', $remainingGedcom, $contMatches, PREG_SET_ORDER)) {
                                        foreach ($contMatches as $contMatch) {
                                            if ($contMatch[1] === 'CONT') {
                                                $fullNote .= "\n" . $contMatch[2];
                                            } else {
                                                $fullNote .= $contMatch[2];
                                            }
                                        }
                                    }
                                }
                                
                                $fullNote = trim($fullNote);
                                if ($fullNote !== '') {
                                    // Erste Zeile der Inline-Notiz extrahieren
                                    $firstLine = strtok($fullNote, "\n");
                                    $firstLine = trim($firstLine);
                                    
                                    // Prueen ob erste Zeile ein Friedhofs-Schluesselwort enthaelt
                                    if (!empty($keywords)) {
                                        foreach ($keywords as $keyword) {
                                            if (stripos($firstLine, $keyword) !== false) {
                                                return ['value' => $firstLine, 'source' => 'NOTE'];
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                    break;
            }
        }
        
        // Fallback: Leerer Wert mit 'NONE' als Quelle
        return ['value' => I18N::translate('Unknown'), 'source' => 'NONE'];
    }

    /**
     * CSV-Export
     */
    public function exportCsvAction(ServerRequestInterface $request): ResponseInterface
    {
        $tree = Validator::attributes($request)->tree();
        $params = (array) $request->getParsedBody();
        
        $place_filter = $params['place_filter'] ?? '';
        $institution_filter = $params['institution_filter'] ?? '';
        $sort_by = $params['sort_by'] ?? 'place';
        
        // Neuer Dropdown-Filter mit Backward-Kompatibilitaet
        $burial_place_filter = $params['burial_place_filter'] ?? 'with_info';
        
        // Backward-Kompatibilitaet: Alte Checkbox-Parameter auf neues System mappen
        if (isset($params['hide_none_filter'])) {
            $burial_place_filter = ($params['hide_none_filter'] === '1') ? 'with_info' : 'all';
        }

        $institutions = $this->collectInstitutions($tree, $place_filter, $institution_filter, $sort_by, $burial_place_filter);

        // CSV erstellen mit gewuenschten Spalten
        $csvContent = "\xEF\xBB\xBF"; // UTF-8 BOM
        $csvContent .= implode(';', [
            I18N::translate('Burial place'),
            I18N::translate('Cemetery/Institution'),
            I18N::translate('Name'),
            I18N::translate('Date of death'),
            I18N::translate('Date of burial'),
            I18N::translate('Address'),
            I18N::translate('Note'),
            I18N::translate('URL'),
        ]) . "\r\n";

        foreach ($institutions as $data) {
            foreach ($data['individuals'] as $entry) {
                $individual = $entry['individual'];
                $burial = $entry['burial'];

                // Sterbedatum ohne HTML-Tags
                $death_fact = $individual->facts(['DEAT'])->first();
                $death_date = $death_fact ? strip_tags($death_fact->date()->display()) : '';
                
                // Bestattungsdatum ohne HTML-Tags
                $burial_date = strip_tags($burial->date()->display());

                // Adresse aus dem BURI-Fakt extrahieren
                $address = '';
                $gedcom = $burial->gedcom();
                if (preg_match('/\n2 ADDR (.+)/', $gedcom, $match)) {
                    $address = trim($match[1]);
                    // CONT und CONC beruecksichtigen
                    if (preg_match_all('/\n3 (CONT|CONC) (.*)/', $gedcom, $contMatches, PREG_SET_ORDER)) {
                        foreach ($contMatches as $contMatch) {
                            if ($contMatch[1] === 'CONT') {
                                $address .= ', ' . trim($contMatch[2]);
                            } else {
                                $address .= trim($contMatch[2]);
                            }
                        }
                    }
                }

                // Notiz aus dem BURI-Fakt extrahieren
                $note = '';
                if (preg_match('/\n2 NOTE (.+)/', $gedcom, $match)) {
                    $noteValue = trim($match[1]);
                    // Pruefen ob es eine Referenz ist (@XREF@)
                    if (preg_match('/@(.+)@/', $noteValue, $refMatch)) {
                        $xref = $refMatch[1];
                        $noteRecord = Registry::noteFactory()->make($xref, $tree);
                        if ($noteRecord !== null) {
                            $note = trim($noteRecord->getNote());
                            // Notiz auf eine Zeile reduzieren
                            $note = str_replace(["\r\n", "\r", "\n"], ' ', $note);
                        }
                    } else {
                        $note = $noteValue;
                    }
                }

                // URL zur Person generieren
                $url = $individual->url();

                $csvContent .= implode(';', [
                    '"' . str_replace('"', '""', $data['place']) . '"',
                    '"' . str_replace('"', '""', $data['institution']) . '"',
                    '"' . str_replace('"', '""', strip_tags($individual->fullName())) . '"',
                    '"' . str_replace('"', '""', $death_date) . '"',
                    '"' . str_replace('"', '""', $burial_date) . '"',
                    '"' . str_replace('"', '""', $address) . '"',
                    '"' . str_replace('"', '""', $note) . '"',
                    '"' . str_replace('"', '""', $url) . '"',
                ]) . "\r\n";
            }
        }

        $filename = 'burial-places-' . $tree->name() . '-' . date('Y-m-d') . '.csv';

        return response($csvContent)
            ->withHeader('Content-Type', 'text/csv; charset=UTF-8')
            ->withHeader('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }
}

return new BurialPlacesReportModule();