# ðŸ“š Technische Dokumentation: Bestattungsorte Modul fÃ¼r Webtrees 2.2

**Version:** 2.2.4.1.2  
**Erstellt:** Oktober 2025  
**Autor:** Thomas Schiller (mit Hilfe von Claude Sonnet 4.5)  
**Modul:** Burial Places Report (Bestattungsorte)  
**Webtrees-Version:** 2.2.1+  
**PHP-Version:** 8.0+

---

## ðŸ“‘ Inhaltsverzeichnis

### Teil 1: Modul-Architektur
1. [Ãœberblick und Modulinformationen](#1-Ã¼berblick-und-modulinformationen)
2. [Architektur und Design-Pattern](#2-architektur-und-design-pattern)
3. [Dateistruktur](#3-dateistruktur)
4. [Implementierte Interfaces und Traits](#4-implementierte-interfaces-und-traits)

### Teil 2: Hauptkomponenten
5. [Haupt-Modul (module.php)](#5-haupt-modul-modulephp)
6. [Hilfsklasse (module_helper.php)](#6-hilfsklasse-module_helperphp)
7. [View Templates](#7-view-templates)
8. [Sprachdateien und I18N](#8-sprachdateien-und-i18n)

### Teil 3: Icon-System (Detailliert)
9. [Icon-System: Konzept und Architektur](#9-icon-system-konzept-und-architektur)
10. [CSS-Struktur und Varianten](#10-css-struktur-und-varianten)
11. [Theme-spezifische Implementierung](#11-theme-spezifische-implementierung)
12. [PHP-Integration: headContent()](#12-php-integration-headcontent)
13. [CSS-Parameter im Detail](#13-css-parameter-im-detail)
14. [Icon-Dateien und Formate](#14-icon-dateien-und-formate)
15. [Fallback-Mechanismus](#15-fallback-mechanismus)

### Teil 4: Datenverarbeitung
16. [Datenfluss und Verarbeitung](#16-datenfluss-und-verarbeitung)
17. [GEDCOM-Verarbeitung](#17-gedcom-verarbeitung)
18. [Datenbank-Interaktion](#18-datenbank-interaktion)
19. [CSV-Export-FunktionalitÃ¤t](#19-csv-export-funktionalitÃ¤t)

### Teil 5: Technische Details
20. [Sicherheit und Validierung](#20-sicherheit-und-validierung)
21. [Performance-Optimierung](#21-performance-optimierung)
22. [Error Handling](#22-error-handling)
23. [Troubleshooting Guide](#23-troubleshooting-guide)

### Teil 6: Erweiterung und Wartung
24. [Erweiterbarkeit](#24-erweiterbarkeit)
25. [Wartung und Updates](#25-wartung-und-updates)
26. [Best Practices](#26-best-practices)
27. [Code-Beispiele](#27-code-beispiele)

### AnhÃ¤nge
- [Anhang A: VollstÃ¤ndiger Code headContent()](#anhang-a-vollstÃ¤ndiger-code-headcontent)
- [Anhang B: Komplette CSS-Datei Beispiel](#anhang-b-komplette-css-datei-beispiel)
- [Anhang C: Performance-Benchmarks](#anhang-c-performance-benchmarks)
- [Anhang D: KompatibilitÃ¤ts-Matrix](#anhang-d-kompatibilitÃ¤ts-matrix)
- [Changelog](#changelog)

---

# Teil 1: Modul-Architektur

## 1. Ãœberblick und Modulinformationen

### 1.1 Modulbeschreibung

Das **Bestattungsorte Modul** ist ein Custom-Modul fÃ¼r Webtrees 2.2+, das Bestattungsereignisse (BURI) aus GEDCOM-Daten intelligent auswertet und nach FriedhÃ¶fen und Institutionen gruppiert darstellt.

**Kernfunktionen:**
- âœ… Intelligente Auswertung von GEDCOM-Feldern (AGNC, PLAC, NOTE)
- âœ… Konfigurierbare Fallback-Logik
- âœ… Keyword-basierte Friedhofserkennung
- âœ… Flexible Filterung und Sortierung
- âœ… CSV-Export mit vollstÃ¤ndigen Daten
- âœ… Theme-spezifische Icon-Darstellung
- âœ… Mehrsprachige UnterstÃ¼tzung (DE, EN, FR, HU)

### 1.2 Technische Spezifikationen

```php
Namespace:          MyVendor\Webtrees\Module\BurialPlacesReport
Hauptklasse:        BurialPlacesReportModule
Version:            2.2.4.1.2
Webtrees-Version:   2.2.1 - 2.2.4
PHP-Version:        8.0 - 8.3
Lizenz:             GPL v3.0
```

### 1.3 Konstanten

```php
// Versionsinformationen
public const CUSTOM_VERSION = '2.2.4.1.2';
public const CUSTOM_WEBSITE = 'https://wt-module.schitho.at';

// Standard-Einstellungen
private const DEFAULT_FALLBACK_ORDER = 'AGNC,PLAC,NOTE';
private const DEFAULT_PLAC_KEYWORDS = 'Friedhof,Cemetery,CimetiÃ¨re,TemetÅ‘';
private const DEFAULT_DIAGNOSIS_MODE = '0';
private const DEFAULT_SHOW_AGNC_ALWAYS = '0';

// Validierungskonstanten
private const MAX_KEYWORDS = 20;              // Maximale Anzahl Keywords
private const MAX_KEYWORD_LENGTH = 50;        // Maximale LÃ¤nge pro Keyword
private const MAX_TOTAL_LENGTH = 250;         // Maximale GesamtlÃ¤nge
```

---

## 2. Architektur und Design-Pattern

### 2.1 MVC-Pattern

Das Modul folgt strikt dem Model-View-Controller Pattern von Webtrees:

```
â”Œâ”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”
â”‚                        CONTROLLER                            â”‚
â”‚                      (module.php)                            â”‚
â”‚  â”Œâ”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”   â”‚
â”‚  â”‚ â€¢ GET/POST Request Handling                          â”‚   â”‚
â”‚  â”‚ â€¢ Route Management                                   â”‚   â”‚
â”‚  â”‚ â€¢ Admin-Konfiguration                                â”‚   â”‚
â”‚  â”‚ â€¢ Data Processing & Validation                       â”‚   â”‚
â”‚  â”‚ â€¢ CSV Export                                         â”‚   â”‚
â”‚  â”‚ â€¢ Icon/CSS Management (headContent)                  â”‚   â”‚
â”‚  â””â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”˜   â”‚
â””â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”¬â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”¬â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”˜
             â”‚                          â”‚
             â”‚ Verwendet                â”‚ Ruft auf
             â–¼                          â–¼
â”Œâ”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”   â”Œâ”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”
â”‚        MODEL           â”‚   â”‚           VIEW                 â”‚
â”‚  (module_helper.php)   â”‚   â”‚   (resources/views/*.phtml)    â”‚
â”‚  â”Œâ”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”  â”‚   â”‚  â”Œâ”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”  â”‚
â”‚  â”‚ NoteFormatter    â”‚  â”‚   â”‚  â”‚ admin.phtml              â”‚  â”‚
â”‚  â”‚ (Markdownâ†’HTML)  â”‚  â”‚   â”‚  â”‚ list-page.phtml          â”‚  â”‚
â”‚  â””â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”˜  â”‚   â”‚  â”‚ admin-tree-select.phtml  â”‚  â”‚
â”‚                        â”‚   â”‚  â”‚ agnc-always-show.phtml   â”‚  â”‚
â””â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”˜   â”‚  â””â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”˜  â”‚
                             â””â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”˜
                                          â”‚
                                          â”‚ Nutzt
                                          â–¼
                             â”Œâ”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”
                             â”‚      RESOURCES                 â”‚
                             â”‚  â”Œâ”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”  â”‚
                             â”‚  â”‚ CSS (Icon-Styles)        â”‚  â”‚
                             â”‚  â”‚ Icons (SVG/PNG)          â”‚  â”‚
                             â”‚  â”‚ Lang (I18N-Dateien)      â”‚  â”‚
                             â”‚  â””â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”˜  â”‚
                             â””â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”€â”˜
```

### 2.2 Dependency Injection

Webtrees nutzt Laravels Service Container fÃ¼r Dependency Injection:

```php
use Fisharebest\Webtrees\Registry;
use Psr\Http\Message\ServerRequestInterface;

// Request aus Container holen
$request = Registry::container()->get(ServerRequestInterface::class);

// Tree-Objekt aus Request extrahieren
$tree = Validator::attributes($request)->tree();
```

### 2.3 Routing-System

```php
public function boot(): void
{
    // View-Namespace registrieren
    View::registerNamespace($this->name(), $this->resourcesFolder() . 'views/');
    
    // Route fÃ¼r Listenseite (GET + POST)
    Registry::routeFactory()->routeMap()
        ->get('burial-places-list', '/tree/{tree}/burial-places', $this)
        ->allows('POST');
    
    // Route fÃ¼r CSV-Export (POST)
    Registry::routeFactory()->routeMap()
        ->post('burial-places-csv', '/tree/{tree}/burial-places/export', 
               [$this, 'exportCsvAction']);
}
```

**Route-Komponenten:**
- `burial-places-list` - Eindeutiger Route-Name
- `/tree/{tree}/burial-places` - URL-Pattern mit Baum-Parameter
- `$this` - Controller (automatisch `handle()` Methode aufrufen)
- `allows('POST')` - Erlaubt auch POST-Requests

---

## 3. Dateistruktur

### 3.1 Komplette Ordnerstruktur

```
burial-places-report/
â”‚
â”œâ”€â”€ module.php                      # Hauptmodul (871 Zeilen)
â”œâ”€â”€ module_helper.php               # Hilfsklassen (47 Zeilen)
â”œâ”€â”€ README.md                       # Benutzer-Dokumentation
â”œâ”€â”€ LICENSE                         # GPL v3.0
â”‚
â””â”€â”€ resources/
    â”‚
    â”œâ”€â”€ css/                        # Icon-Stylesheets
    â”‚   â”œâ”€â”€ menu-icons.webtrees.css       # Webtrees-Theme
    â”‚   â”œâ”€â”€ menu-icons.colors.css         # Colors-Theme
    â”‚   â”œâ”€â”€ menu-icons.fab.css            # FAB-Theme
    â”‚   â”œâ”€â”€ menu-icons.minimal.css        # Minimal-Theme
    â”‚   â”œâ”€â”€ menu-icons.clouds.css         # Clouds-Theme
    â”‚   â”œâ”€â”€ menu-icons.xenea.css          # Xenea-Theme
    â”‚   â””â”€â”€ menu-icons.universal.css      # Fallback fÃ¼r alle Themes
    â”‚
    â”œâ”€â”€ icons/                      # Icon-Dateien
    â”‚   â”œâ”€â”€ menu-list-burial-webtrees.svg
    â”‚   â”œâ”€â”€ menu-list-burial-colors.svg
    â”‚   â”œâ”€â”€ menu-list-burial-fab.svg
    â”‚   â”œâ”€â”€ menu-list-burial-minimal.svg
    â”‚   â”œâ”€â”€ menu-list-burial-clouds.svg
    â”‚   â”œâ”€â”€ menu-list-burial-xenea.svg
    â”‚   â””â”€â”€ menu-list-burial-universal.svg
    â”‚
    â”œâ”€â”€ lang/                       # Ãœbersetzungen
    â”‚   â”œâ”€â”€ de.php                  # Deutsch (328 Zeilen)
    â”‚   â”œâ”€â”€ en-GB.php               # Englisch (328 Zeilen)
    â”‚   â”œâ”€â”€ fr.php                  # FranzÃ¶sisch (329 Zeilen)
    â”‚   â””â”€â”€ hu.php                  # Ungarisch (330 Zeilen)
    â”‚
    â””â”€â”€ views/                      # View-Templates
        â”œâ”€â”€ admin.phtml             # Admin-Konfiguration (357 Zeilen)
        â”œâ”€â”€ admin-tree-select.phtml # Baum-Auswahl (45 Zeilen)
        â”œâ”€â”€ list-page.phtml         # Hauptliste (385 Zeilen)
        â””â”€â”€ agnc-always-show.phtml  # JavaScript fÃ¼r AGNC-Feld (95 Zeilen)
```

### 3.2 DateigrÃ¶ÃŸen

```
PHP-Dateien:
  module.php                 31 KB
  module_helper.php           1 KB

View-Templates:
  admin.phtml                14 KB
  admin-tree-select.phtml     1.5 KB
  list-page.phtml            17 KB
  agnc-always-show.phtml     14 KB

CSS-Dateien:
  menu-icons.*.css          ~6 KB pro Datei
  Gesamt CSS                ~42 KB

Icon-Dateien:
  *.svg                     ~2-5 KB pro Icon
  Gesamt Icons              ~28 KB

Sprachdateien:
  de.php                     9.5 KB
  en-GB.php                  8.5 KB
  fr.php                    10 KB
  hu.php                    11 KB
  Gesamt Sprachen           39 KB

GESAMT MODUL:              ~187 KB
```

---

## 4. Implementierte Interfaces und Traits

### 4.1 Interfaces

```php
class BurialPlacesReportModule extends AbstractModule implements 
    ModuleCustomInterface,      // Custom-Modul
    ModuleListInterface,        // Listenseite im MenÃ¼
    ModuleConfigInterface,      // Konfigurationsseite
    ModuleGlobalInterface       // Globale FunktionalitÃ¤t (JS/CSS)
{
    // ...
}
```

**Interface-Beschreibungen:**

#### ModuleCustomInterface
```php
// Pflicht-Methoden:
public function customModuleAuthorName(): string;
public function customModuleSupportUrl(): string;
public function customModuleLatestVersionUrl(): string;
public function customModuleVersion(): string;
```

#### ModuleListInterface
```php
// Pflicht-Methoden:
public function listUrl(Tree $tree, array $parameters = []): string;
public function listUrlAttributes(): array;
public function listIsEmpty(Tree $tree): bool;

// Optional:
public function listMenuClass(): string;  // CSS-Klasse fÃ¼r MenÃ¼
```

#### ModuleConfigInterface
```php
// Pflicht-Methoden:
public function getConfigLink(): string;
```

#### ModuleGlobalInterface
```php
// Optional, aber wichtig fÃ¼r Icons:
public function headContent(): string;     // CSS/JS in <head>
public function bodyContent(): string;     // HTML vor </body>
```

### 4.2 Traits

```php
use ModuleCustomTrait;
use ModuleListTrait;
use ModuleConfigTrait;
use ModuleGlobalTrait;
```

**Trait-FunktionalitÃ¤t:**
- Stellen Standard-Implementierungen bereit
- KÃ¶nnen Ã¼berschrieben werden
- `ModuleGlobalTrait` aktiviert `headContent()` und `bodyContent()`

---

# Teil 2: Hauptkomponenten

## 5. Haupt-Modul (module.php)

### 5.1 Klassendefinition

```php
<?php

declare(strict_types=1);

namespace MyVendor\Webtrees\Module\BurialPlacesReport;

use Fisharebest\Webtrees\Auth;
use Fisharebest\Webtrees\FlashMessages;
use Fisharebest\Webtrees\I18N;
use Fisharebest\Webtrees\Module\AbstractModule;
use Fisharebest\Webtrees\Module\ModuleConfigInterface;
use Fisharebest\Webtrees\Module\ModuleConfigTrait;
use Fisharebest\Webtrees\Module\ModuleCustomInterface;
use Fisharebest\Webtrees\Module\ModuleCustomTrait;
use Fisharebest\Webtrees\Module\ModuleGlobalInterface;
use Fisharebest\Webtrees\Module\ModuleGlobalTrait;
use Fisharebest\Webtrees\Module\ModuleListInterface;
use Fisharebest\Webtrees\Module\ModuleListTrait;
use Fisharebest\Webtrees\Registry;
use Fisharebest\Webtrees\Tree;
use Fisharebest\Webtrees\Validator;
use Fisharebest\Webtrees\View;
use Illuminate\Database\Capsule\Manager as DB;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

require_once __DIR__ . '/module_helper.php';

class BurialPlacesReportModule extends AbstractModule implements 
    ModuleCustomInterface,
    ModuleListInterface,
    ModuleConfigInterface,
    ModuleGlobalInterface
{
    use ModuleCustomTrait;
    use ModuleListTrait;
    use ModuleConfigTrait;
    use ModuleGlobalTrait;

    // Konstanten siehe Abschnitt 1.3
}
```

### 5.2 Kern-Methoden Ãœbersicht

| Methode | Zweck | RÃ¼ckgabe |
|---------|-------|----------|
| `boot()` | Modul-Initialisierung | `void` |
| `title()` | Modul-Titel | `string` |
| `description()` | Modul-Beschreibung | `string` |
| `listMenuClass()` | CSS-Klasse fÃ¼r MenÃ¼ | `string` |
| `headContent()` | CSS/JS in `<head>` | `string` |
| `getConfigLink()` | Link zur Admin-Seite | `string` |
| `getAdminAction()` | GET Admin-Request | `ResponseInterface` |
| `postAdminAction()` | POST Admin-Request | `ResponseInterface` |
| `handle()` | GET/POST Hauptseite | `ResponseInterface` |
| `exportCsvAction()` | CSV-Export | `ResponseInterface` |

### 5.3 boot() - Initialisierung

**Funktion:** Wird beim Modul-Start automatisch aufgerufen.

```php
public function boot(): void
{
    // 1. View-Namespace registrieren
    View::registerNamespace(
        $this->name(),                           // Modul-Name
        $this->resourcesFolder() . 'views/'      // Pfad zu Views
    );
    
    // 2. Route fÃ¼r Listenseite (GET + POST)
    Registry::routeFactory()->routeMap()
        ->get('burial-places-list', '/tree/{tree}/burial-places', $this)
        ->allows('POST');
    
    // 3. Route fÃ¼r CSV-Export (POST)
    Registry::routeFactory()->routeMap()
        ->post('burial-places-csv', '/tree/{tree}/burial-places/export', 
               [$this, 'exportCsvAction']);
}
```

**Wichtige Punkte:**
- `View::registerNamespace()` ermÃ¶glicht `view('module-name::template')`
- `->allows('POST')` macht GET-Route auch fÃ¼r POST verfÃ¼gbar
- CSV-Export benÃ¶tigt eigene POST-Route fÃ¼r saubere Trennung

### 5.4 listMenuClass() - CSS-Klasse fÃ¼r MenÃ¼

```php
public function listMenuClass(): string
{
    return 'menu-list-burial';
}
```

**HTML-Output:**
```html
<li class="nav-item dropdown">
    <a class="nav-link dropdown-toggle" ...>Listen</a>
    <div class="dropdown-menu">
        <a class="dropdown-item menu-list-burial" href="...">
            Bestattungsorte
        </a>
    </div>
</li>
```

Diese Klasse wird von den CSS-Dateien verwendet, um Icons hinzuzufÃ¼gen (siehe Teil 3).

### 5.5 Hilfsmethoden fÃ¼r Preferences

```php
/**
 * Konsistentes Lesen von baum-spezifischen Einstellungen
 */
private function getTreePreference(Tree $tree, string $key, string $default): string
{
    $full_key = 'burial_places_' . $key;
    return $tree->getPreference($full_key, $default);
}

/**
 * Konsistentes Schreiben von baum-spezifischen Einstellungen
 */
private function setTreePreference(Tree $tree, string $key, string $value): void
{
    $full_key = 'burial_places_' . $key;
    $tree->setPreference($full_key, $value);
}
```

**Vorteile:**
- Zentrale Verwaltung der Preference-Namen
- Kein Tippfehler bei `burial_places_` PrÃ¤fix
- Einfache Ã„nderung des PrÃ¤fixes mÃ¶glich

**Verwendung:**
```php
// Lesen
$fallback_order = $this->getTreePreference($tree, 'fallback_order', self::DEFAULT_FALLBACK_ORDER);

// Schreiben
$this->setTreePreference($tree, 'fallback_order', 'AGNC,NOTE,PLAC');
```

### 5.6 validateKeywords() - Keyword-Validierung

**Zweck:** Validiert und bereinigt die Keyword-Eingabe fÃ¼r Friedhofserkennung.

```php
/**
 * Validiert und bereinigt Keyword-Eingabe
 * 
 * @param string $input Rohe Eingabe (kommagetrennt, ZeilenumbrÃ¼che, etc.)
 * @return array ['valid' => bool, 'error' => string, 'keywords' => string]
 */
private function validateKeywords(string $input): array
{
    // 1. Trennen bei Komma, Semikolon oder Zeilenumbruch
    $keywords = preg_split('/[,;\n\r]+/', $input);
    
    // 2. Whitespace entfernen
    $keywords = array_map('trim', $keywords);
    
    // 3. Leere EintrÃ¤ge entfernen
    $keywords = array_filter($keywords, fn($k) => $k !== '');
    
    // 4. Duplikate entfernen (case-insensitive)
    $unique_keywords = [];
    $seen = [];
    foreach ($keywords as $keyword) {
        $lower = mb_strtolower($keyword);
        if (!isset($seen[$lower])) {
            $unique_keywords[] = $keyword;
            $seen[$lower] = true;
        }
    }
    $keywords = $unique_keywords;
    
    // 5. Validierung: Anzahl
    if (count($keywords) > self::MAX_KEYWORDS) {
        return [
            'valid' => false,
            'error' => I18N::translate('Too many keywords. Maximum: %s', self::MAX_KEYWORDS),
            'keywords' => '',
        ];
    }
    
    // 6. Validierung: LÃ¤nge pro Keyword
    foreach ($keywords as $keyword) {
        if (mb_strlen($keyword) > self::MAX_KEYWORD_LENGTH) {
            return [
                'valid' => false,
                'error' => I18N::translate('Keyword too long: "%s". Maximum: %s characters', 
                                          $keyword, self::MAX_KEYWORD_LENGTH),
                'keywords' => '',
            ];
        }
    }
    
    // 7. Validierung: GesamtlÃ¤nge
    $cleaned = implode(',', $keywords);
    if (mb_strlen($cleaned) > self::MAX_TOTAL_LENGTH) {
        return [
            'valid' => false,
            'error' => I18N::translate('Total length too long. Maximum: %s characters', 
                                      self::MAX_TOTAL_LENGTH),
            'keywords' => '',
        ];
    }
    
    // 8. Erfolg
    return [
        'valid' => true,
        'error' => '',
        'keywords' => $cleaned,
    ];
}
```

**Validierungsregeln:**
- Max. 20 Keywords
- Max. 50 Zeichen pro Keyword
- Max. 250 Zeichen Gesamt (Sicherheitspuffer fÃ¼r VARCHAR(255))
- Duplikate werden entfernt (case-insensitive)
- Mehrere Trennzeichen werden unterstÃ¼tzt (`,`, `;`, Zeilenumbruch)

**Beispiele:**

```php
// Eingabe: "Friedhof, Cemetery;CimetiÃ¨re\nTemetÅ‘"
// Output:  "Friedhof,Cemetery,CimetiÃ¨re,TemetÅ‘"

// Eingabe: "Friedhof, friedhof, FRIEDHOF"
// Output:  "Friedhof"  (Duplikate entfernt)

// Eingabe: "Friedhof,   Cemetery  ,   CimetiÃ¨re"
// Output:  "Friedhof,Cemetery,CimetiÃ¨re"  (Whitespace entfernt)
```

---

## 6. Hilfsklasse (module_helper.php)

### 6.1 NoteFormatter Klasse

**Zweck:** Konvertiert Markdown-Ã¤hnliche GEDCOM-Notizen in HTML.

```php
<?php

declare(strict_types=1);

namespace MyVendor\Webtrees\Module\BurialPlacesReport;

class NoteFormatter
{
    /**
     * Wandelt Markdown-Ã¤hnliche Syntax in HTML um
     * 
     * UnterstÃ¼tzte Syntax:
     * - **fett** â†’ <strong>fett</strong>
     * - *kursiv* â†’ <em>kursiv</em>
     * 
     * @param string $text Roher Text
     * @return string HTML-formatierter Text
     */
    public static function format(string $text): string
    {
        // 1. HTML-Escape zur Sicherheit
        $text = htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
        
        // 2. **fett** â†’ <strong>fett</strong>
        $text = preg_replace(
            '/\*\*(.+?)\*\*/',      // Pattern
            '<strong>$1</strong>',   // Replacement
            $text
        );
        
        // 3. *kursiv* â†’ <em>kursiv</em>
        $text = preg_replace(
            '/\*(.+?)\*/',           // Pattern
            '<em>$1</em>',           // Replacement
            $text
        );
        
        // 4. ZeilenumbrÃ¼che â†’ <br>
        $text = nl2br($text);
        
        return $text;
    }
}
```

**Verwendung:**
```php
$note_text = "Dies ist **wichtig** und *interessant*.";
$html = NoteFormatter::format($note_text);
// Output: "Dies ist <strong>wichtig</strong> und <em>interessant</em>."
```

**Sicherheit:**
- `htmlspecialchars()` verhindert XSS-Angriffe
- Nur sichere HTML-Tags werden generiert (`<strong>`, `<em>`, `<br>`)
- Keine Benutzereingabe wird direkt als HTML ausgegeben

---

## 7. View Templates

### 7.1 Ãœbersicht

| Template | Zweck | Zeilen |
|----------|-------|--------|
| `admin.phtml` | Konfigurations-OberflÃ¤che | 357 |
| `admin-tree-select.phtml` | Baum-Auswahl | 45 |
| `list-page.phtml` | Hauptliste | 385 |
| `agnc-always-show.phtml` | JavaScript fÃ¼r AGNC | 95 |

### 7.2 admin.phtml - Konfiguration

**Struktur:**
```php
<?php
/**
 * @var string $title         Modul-Titel
 * @var Tree   $tree          Aktueller Baum
 * @var string $fallback_order    AGNC,PLAC,NOTE
 * @var string $plac_keywords     Keyword-Liste
 * @var string $diagnosis_mode    0/1
 * @var string $show_agnc_always  0/1
 */

use Fisharebest\Webtrees\I18N;
use Fisharebest\Webtrees\View;
?>

<!-- Header -->
<h1><?= e($title) ?></h1>

<!-- Baum-Info -->
<div class="alert alert-info">
    <?= I18N::translate('Current tree: %s', $tree->title()) ?>
</div>

<!-- Formular -->
<form method="post">
    <?= csrf_field() ?>
    
    <!-- 1. AGNC-Feld immer anzeigen -->
    <div class="form-group">
        <label>
            <input type="checkbox" name="show_agnc_always" value="1" 
                   <?= $show_agnc_always === '1' ? 'checked' : '' ?>>
            <?= I18N::translate('Always show cemetery field (Institution) for burials') ?>
        </label>
    </div>
    
    <!-- 2. Fallback-Reihenfolge (Drag & Drop) -->
    <div class="form-group">
        <label><?= I18N::translate('Order for value determination') ?></label>
        <div id="sortable-fallback">
            <?php foreach (explode(',', $fallback_order) as $source): ?>
                <div class="sortable-item" data-source="<?= e($source) ?>">
                    <span class="drag-handle">â‹®â‹®</span>
                    <?= e($source) ?>
                </div>
            <?php endforeach ?>
        </div>
        <input type="hidden" name="fallback_order" id="fallback_order_input" 
               value="<?= e($fallback_order) ?>">
    </div>
    
    <!-- 3. Keywords -->
    <div class="form-group">
        <label for="plac_keywords">
            <?= I18N::translate('Cemetery keywords') ?>
        </label>
        <textarea name="plac_keywords" id="plac_keywords" 
                  class="form-control" rows="3"><?= e($plac_keywords) ?></textarea>
        <small class="form-text text-muted">
            <?= I18N::translate('Comma-separated list') ?>
        </small>
    </div>
    
    <!-- 4. Diagnosemodus -->
    <div class="form-group">
        <label>
            <input type="checkbox" name="diagnosis_mode" value="1" 
                   <?= $diagnosis_mode === '1' ? 'checked' : '' ?>>
            <?= I18N::translate('Show source tag (AGNC/NOTE/PLAC) in parentheses') ?>
        </label>
    </div>
    
    <!-- Buttons -->
    <div class="form-group">
        <button type="submit" class="btn btn-primary">
            <?= I18N::translate('Save') ?>
        </button>
        <a href="..." class="btn btn-secondary">
            <?= I18N::translate('Reset to default values') ?>
        </a>
    </div>
</form>

<!-- JavaScript fÃ¼r Sortable -->
<script>
// jQuery UI Sortable fÃ¼r Drag & Drop
$('#sortable-fallback').sortable({
    handle: '.drag-handle',
    update: function() {
        var order = [];
        $(this).find('.sortable-item').each(function() {
            order.push($(this).data('source'));
        });
        $('#fallback_order_input').val(order.join(','));
    }
});
</script>
```

**Features:**
- **Checkbox-Inputs** fÃ¼r boolean Optionen
- **Drag & Drop** fÃ¼r Fallback-Reihenfolge (jQuery UI Sortable)
- **Textarea** fÃ¼r Keywords
- **Hidden Input** fÃ¼r serialisierte Reihenfolge
- **CSRF-Token** fÃ¼r Sicherheit

### 7.3 list-page.phtml - Hauptliste

**Struktur:**
```php
<?php
/**
 * @var string $title
 * @var array  $institutions  Gruppierte Daten
 * @var string $filter_place
 * @var string $filter_institution
 * @var string $sort_by
 * @var bool   $diagnosis_mode
 */
?>

<h1><?= e($title) ?></h1>

<!-- Filter-Formular -->
<form method="post" class="mb-3">
    <?= csrf_field() ?>
    
    <div class="row">
        <!-- Bestattungsort-Filter -->
        <div class="col-md-4">
            <label><?= I18N::translate('Burial place') ?></label>
            <input type="text" name="filter_place" class="form-control" 
                   value="<?= e($filter_place) ?>">
        </div>
        
        <!-- Institution-Filter -->
        <div class="col-md-4">
            <label><?= I18N::translate('Cemetery/Institution') ?></label>
            <input type="text" name="filter_institution" class="form-control" 
                   value="<?= e($filter_institution) ?>">
        </div>
        
        <!-- Sortierung -->
        <div class="col-md-4">
            <label><?= I18N::translate('Sort by') ?></label>
            <select name="sort_by" class="form-control">
                <option value="place" <?= $sort_by === 'place' ? 'selected' : '' ?>>
                    <?= I18N::translate('Burial place') ?>
                </option>
                <option value="institution" <?= $sort_by === 'institution' ? 'selected' : '' ?>>
                    <?= I18N::translate('Cemetery/Institution') ?>
                </option>
            </select>
        </div>
    </div>
    
    <div class="mt-2">
        <button type="submit" class="btn btn-primary">
            <?= I18N::translate('Generate list') ?>
        </button>
        <button type="submit" name="export_csv" value="1" class="btn btn-success">
            <?= I18N::translate('Download as CSV') ?>
        </button>
    </div>
</form>

<!-- Ergebnisse -->
<div class="results">
    <?php foreach ($institutions as $data): ?>
        <div class="institution-group">
            <h2>
                <?= e($data['place']) ?>
                <?php if ($data['institution']): ?>
                    â€º <?= e($data['institution']) ?>
                    <?php if ($diagnosis_mode && isset($data['source'])): ?>
                        <small class="text-muted">(<?= e($data['source']) ?>)</small>
                    <?php endif ?>
                <?php endif ?>
                <span class="badge badge-secondary"><?= count($data['individuals']) ?></span>
            </h2>
            
            <table class="table table-sm">
                <thead>
                    <tr>
                        <th><?= I18N::translate('Name') ?></th>
                        <th><?= I18N::translate('Birth') ?></th>
                        <th><?= I18N::translate('Death') ?></th>
                        <th><?= I18N::translate('Burial') ?></th>
                        <th><?= I18N::translate('Age') ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($data['individuals'] as $entry): ?>
                        <tr>
                            <td>
                                <a href="<?= e($entry['individual']->url()) ?>">
                                    <?= e($entry['individual']->fullName()) ?>
                                </a>
                            </td>
                            <td><?= $entry['birth_date'] ?></td>
                            <td><?= $entry['death_date'] ?></td>
                            <td><?= $entry['burial_date'] ?></td>
                            <td><?= $entry['age'] ?></td>
                        </tr>
                    <?php endforeach ?>
                </tbody>
            </table>
        </div>
    <?php endforeach ?>
</div>
```

**Features:**
- **Responsive Design** (Bootstrap Grid)
- **Filter-Formular** mit mehreren Feldern
- **Sortier-Option** (Dropdown)
- **CSV-Export-Button** (separate POST-Action)
- **Gruppierte Darstellung** (verschachtelte Loops)
- **Diagnosemodus** (zeigt Datenquelle)

### 7.4 agnc-always-show.phtml - AGNC JavaScript

**Zweck:** Zeigt AGNC-Feld (Institution) im Bestattungs-Editor immer an.

```php
<?php
/**
 * @var string $module_name
 */
?>

<script>
(function() {
    'use strict';
    
    /**
     * PrÃ¼ft ob AGNC-Option aktiviert ist
     */
    function isAGNCAlwaysShowEnabled() {
        // AJAX-Request an Modul-Einstellung
        return <?= $show_agnc_always === '1' ? 'true' : 'false' ?>;
    }
    
    /**
     * Manipuliert Bestattungs-Formular
     */
    function enhanceBurialForm() {
        if (!isAGNCAlwaysShowEnabled()) {
            return;
        }
        
        // Warte auf Formular-Initialisierung
        var checkInterval = setInterval(function() {
            var burialForm = document.querySelector('form[data-wt-fact="BURI"]');
            
            if (burialForm) {
                clearInterval(checkInterval);
                
                // Suche AGNC-Feld
                var agncField = burialForm.querySelector('input[name$="[AGNC]"]');
                var agncRow = agncField ? agncField.closest('.form-group, .row') : null;
                
                if (agncRow) {
                    // Zeige Feld immer (auch wenn leer)
                    agncRow.style.display = 'block';
                    
                    // FÃ¼ge Hinweis hinzu
                    var label = agncRow.querySelector('label');
                    if (label && !label.querySelector('.text-muted')) {
                        var hint = document.createElement('small');
                        hint.className = 'text-muted ml-2';
                        hint.textContent = '(<?= I18N::translate('Cemetery/Institution') ?>)';
                        label.appendChild(hint);
                    }
                }
            }
        }, 500);
        
        // Timeout nach 10 Sekunden
        setTimeout(function() {
            clearInterval(checkInterval);
        }, 10000);
    }
    
    // Initialisierung
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', enhanceBurialForm);
    } else {
        enhanceBurialForm();
    }
})();
</script>
```

**Funktionsweise:**
1. PrÃ¼ft ob Option aktiviert ist
2. Wartet auf Formular-Initialisierung (Polling)
3. Sucht AGNC-Feld im Bestattungs-Formular
4. Zeigt Feld immer an (`display: block`)
5. FÃ¼gt hilfreichen Hinweis hinzu
6. Timeout nach 10 Sekunden

---

## 8. Sprachdateien und I18N

### 8.1 Struktur

```php
<?php

return [
    'Burial Places Report' => 'Bestattungsorte',
    'A list of burials...' => 'Eine Liste von Bestattungen...',
    
    // Admin-Strings
    'Cemetery keywords' => 'Friedhofs-SchlÃ¼sselwÃ¶rter',
    'Order for value determination' => 'Reihenfolge zur Werteermittlung',
    
    // Filter-Strings
    'Burial place' => 'Bestattungsort',
    'Cemetery/Institution' => 'Friedhof/Institution',
    
    // Platzhalter mit Parametern
    'Too many keywords. Maximum: %s' => 'Zu viele SchlÃ¼sselwÃ¶rter. Maximum: %s',
    'Current tree: %s' => 'Aktueller Baum: %s',
];
```

### 8.2 Verwendung in Templates

```php
// Einfacher String
<?= I18N::translate('Burial place') ?>

// String mit Parameter
<?= I18N::translate('Current tree: %s', $tree->title()) ?>

// Plural-Form
<?= I18N::plural('%s burial', '%s burials', $count, I18N::number($count)) ?>
```

### 8.3 Best Practices

âœ… **DO:**
- Verwende `I18N::translate()` fÃ¼r alle Benutzer-sichtbaren Strings
- Verwende Platzhalter (`%s`) fÃ¼r dynamische Werte
- Verwende `I18N::plural()` fÃ¼r Mehrzahl-Formen
- Verwende kurze, prÃ¤gnante Keys (nicht ganze SÃ¤tze)

âŒ **DON'T:**
- String-Konkatenation fÃ¼r Ã¼bersetzte Texte
- HTML in Ãœbersetzungs-Strings
- Zu lange Keys (max. ~50 Zeichen)
- Hardcoded Strings im Code

---

# Teil 3: Icon-System (Detailliert)

## 9. Icon-System: Konzept und Architektur

### 9.1 Grundprinzip

Das Icon-System basiert auf **CSS-Selektoren mit Theme-spezifischen Klassen**. Webtrees setzt automatisch eine Theme-Klasse am `<body>`-Element:

```html
<!-- Webtrees-Theme -->
<body class="wt-theme-webtrees">
    ...
</body>

<!-- Colors-Theme -->
<body class="wt-theme-colors">
    ...
</body>

<!-- Clouds-Theme -->
<body class="wt-theme-clouds">
    ...
</body>
```

### 9.2 CSS-Pseudo-Element fÃ¼r Icon

Das Icon wird Ã¼ber das **CSS-Pseudo-Element `::before`** eingefÃ¼gt:

```css
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial::before {
    content: "";                                               /* Leeres Pseudo-Element */
    position: absolute;                                         /* Absolute Positionierung */
    background-image: url("../icons/menu-list-burial-webtrees.svg"); /* Icon-Bild */
    background-repeat: no-repeat;                              /* Kein Wiederholen */
    background-size: contain;                                  /* Bild skalieren */
    background-position: center;                               /* Bild zentrieren */
    /* ... weitere Eigenschaften ... */
}
```

### 9.3 HTML-Struktur des MenÃ¼s

```html
<nav class="wt-genealogy-menu">
    <ul class="navbar-nav">
        <li class="nav-item dropdown">
            <a class="nav-link dropdown-toggle">Listen</a>
            <div class="dropdown-menu">
                <!-- Unser MenÃ¼punkt -->
                <a class="dropdown-item menu-list-burial" href="/tree/demo/burial-places">
                    Bestattungsorte
                </a>
            </div>
        </li>
    </ul>
</nav>
```

**CSS-Selektor-Hierarchie:**
```
.wt-theme-webtrees          â† Theme-spezifisch (body-Klasse)
  .wt-genealogy-menu        â† MenÃ¼-Container
    .dropdown-item          â† MenÃ¼punkt
      .menu-list-burial     â† Unser Modul (listMenuClass)
        ::before            â† Icon-Pseudo-Element
```

### 9.4 Warum CSS statt PHP/HTML?

| Methode | Vorteile | Nachteile |
|---------|----------|-----------|
| **CSS-basiert** | âœ… Keine Serverlogik<br>âœ… Theme-individuell<br>âœ… Benutzer-anpassbar<br>âœ… Browser-Cache<br>âœ… Keine Template-Ã„nderungen | âŒ Mehr CSS-Dateien<br>âŒ Komplexere Struktur |
| **PHP/HTML-basiert** | âœ… Weniger Dateien<br>âœ… Direkte Kontrolle | âŒ Theme-Erkennung nÃ¶tig<br>âŒ Nicht anpassbar<br>âŒ Template-Ã„nderungen<br>âŒ Mehr Serverlogik |

**Entscheidung:** CSS-basiert fÃ¼r maximale FlexibilitÃ¤t und Wartbarkeit!

---

## 10. CSS-Struktur und Varianten

### 10.1 Die drei CSS-Varianten

Das System verwendet **drei Arten von CSS-Dateien**:

#### 1ï¸âƒ£ Universal-CSS (`menu-icons.universal.css`)

**Zweck:** Fallback fÃ¼r alle Themes ohne eigene CSS-Datei.

**Eigenschaften:**
- âŒ **Keine Theme-Klasse** im Selektor
- âœ… Funktioniert mit **allen Themes**
- âœ… Generisches Icon
- âœ… Moderate Werte

**CSS-Selektor:**
```css
/* KEIN .wt-theme-xxx PrÃ¤fix! */
.dropdown-item.menu-list-burial {
    position: relative !important;
    line-height: 1.50 !important;
    padding: 0.40rem 1rem 0.50rem 2.20em !important;
}

.dropdown-item.menu-list-burial::before {
    content: "" !important;
    position: absolute;
    left: 0.35em;
    top: 40%;
    /* ... */
    background-image: url("../icons/menu-list-burial-universal.svg");
}
```

**Wichtig:** Diese Datei ist **NICHT geeignet als Vorlage** fÃ¼r theme-spezifische CSS!

---

#### 2ï¸âƒ£ Theme-spezifische CSS mit Icon

**Zweck:** Individuelles Styling fÃ¼r ein bestimmtes Theme.

**Eigenschaften:**
- âœ… **Theme-Klasse** im Selektor
- âœ… Eigenes Icon mÃ¶glich
- âœ… Angepasste Werte
- âœ… Volle Kontrolle

**CSS-Selektor:**
```css
/* Mit .wt-theme-xxx PrÃ¤fix! */
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
    position: relative;
    line-height: 1.90 !important;
    padding: 0rem 1rem 0.70rem 2.20em !important;
}

.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial::before {
    content: "";
    position: absolute;
    left: 0.35em;
    top: 32%;
    /* ... */
    background-image: url("../icons/menu-list-burial-webtrees.svg");
}
```

**VerfÃ¼gbare Theme-CSS-Dateien:**
- `menu-icons.webtrees.css`
- `menu-icons.colors.css`
- `menu-icons.fab.css`
- `menu-icons.minimal.css`
- `menu-icons.clouds.css`
- `menu-icons.xenea.css`

---

#### 3ï¸âƒ£ Theme-spezifische CSS ohne Icon

**Zweck:** MenÃ¼punkt ohne Icon fÃ¼r minimalistische Themes.

**Eigenschaften:**
- âœ… **Theme-Klasse** im Selektor
- âŒ **Kein Icon** (explizit deaktiviert)
- âœ… Reduzierte AbstÃ¤nde
- âœ… Kleinere ZeilenhÃ¶he

**CSS-Selektor:**
```css
.wt-theme-minimal .wt-genealogy-menu .dropdown-item.menu-list-burial {
    position: relative;
    line-height: 1.40 !important;
    padding: 0.40rem 1rem !important;  /* KEIN padding-left fÃ¼r Icon! */
}

.wt-theme-minimal .wt-genealogy-menu .dropdown-item.menu-list-burial::before {
    content: none !important;  /* Kein Icon! */
    display: none !important;
}
```

---

### 10.2 CSS-Datei-Namenskonvention

**Schema:**
```
menu-icons.<theme-name>.css
```

**Beispiele:**
```
menu-icons.webtrees.css     â†’ FÃ¼r .wt-theme-webtrees
menu-icons.colors.css       â†’ FÃ¼r .wt-theme-colors
menu-icons.fab.css          â†’ FÃ¼r .wt-theme-fab
menu-icons.minimal.css      â†’ FÃ¼r .wt-theme-minimal
menu-icons.universal.css    â†’ FÃ¼r alle Themes (kein Theme-Name!)
```

**Wichtig:** Der Theme-Name im Dateinamen muss **exakt** mit der `<body>`-Klasse Ã¼bereinstimmen!

```html
<body class="wt-theme-colors">  <!-- Theme-Name: colors -->
```
â†“
```
menu-icons.colors.css  â† Richtig!
menu-icons.Colors.css  â† FALSCH (GroÃŸ-/Kleinschreibung!)
menu-icons.colour.css  â† FALSCH (Schreibfehler!)
```

---

## 11. Theme-spezifische Implementierung

### 11.1 Neues Theme hinzufÃ¼gen - Schritt fÃ¼r Schritt

#### Schritt 1: Theme-Namen ermitteln

**Methode A: Modul-Debug-Kommentar (EINFACHSTE METHODE!) â­**

Das Modul gibt automatisch einen HTML-Kommentar mit dem erkannten Theme aus:

1. Ã–ffne Webtrees mit dem gewÃ¼nschten Theme
2. Ã–ffne eine beliebige Seite, auf der das Modul aktiv ist
3. Rechtsklick â†’ "Seitenquelltext anzeigen"
4. Suche nach `Detected Theme:` (Strg+F)

**Beispiel:**
```html
<!-- Burial Places Module - Icon CSS -->
<!-- Detected Theme: _jc-theme-justlight_ -->
<!-- Loaded CSS files: menu-icons_justlight_.css -->
<style>
...
</style>
```

**Was Du hier siehst:**
- âœ… **Erkannter Theme-Name:** `_jc-theme-justlight_` (mit Unterstrichen!)
- âœ… **Geladene CSS-Datei:** `menu-icons_justlight_.css`
- âœ… **Ob CSS geladen wurde:** Wenn `<style>` leer ist â†’ Problem!

**Wichtig:**  
Der Theme-Name im Kommentar entspricht **exakt** dem Namen, den das Modul verwendet. Wenn hier `_jc-theme-justlight_` steht, muss die CSS-Datei `menu-icons_justlight_.css` heiÃŸen (mit Unterstrichen statt Bindestrichen!).

---

**Methode B: HTML-Body-Klasse**
1. Ã–ffne Webtrees mit dem gewÃ¼nschten Theme
2. Rechtsklick â†’ "Seitenquelltext anzeigen"
3. Suche nach `<body class="`
4. Notiere die Theme-Klasse

```html
<body class="wt-theme-mein-neues-theme">
```
â†“ Theme-Name: `mein-neues-theme`

**Methode C: Browser-Konsole**
1. DrÃ¼cke `F12` (DevTools Ã¶ffnen)
2. Console-Tab
3. Eingabe: `document.body.className`
4. Output: `wt-theme-mein-neues-theme`

**Methode D: PHP-Code (fÃ¼r Entwickler)**
```php
$theme = Registry::container()->get(ServerRequestInterface::class)
                              ->getAttribute('theme');
echo $theme->name();  // Output: "mein-neues-theme"
```

---

#### Schritt 2: CSS-Datei erstellen

**Datei anlegen:**
```
/modules_v4/burial-places-report/resources/css/menu-icons.mein-neues-theme.css
```

**Vorlage verwenden:**
```bash
# Kopiere eine bestehende theme-spezifische CSS
cp menu-icons.webtrees.css menu-icons.mein-neues-theme.css
```

**âš ï¸ WICHTIG:** Verwende **NICHT** `menu-icons.universal.css` als Vorlage!

---

#### Schritt 3: CSS-Datei anpassen

**Header Ã¤ndern:**
```css
/* ============================================================
   mein-neues-theme â€“ MenÃ¼punkt "Bestattungsorte"
   ============================================================
   Diese Datei wirkt NUR, wenn das mein-neues-theme aktiv ist.
   ============================================================ */
```

**Theme-Klasse ersetzen:**

Suche & Ersetze in der gesamten Datei:
- **Suche:** `.wt-theme-webtrees`
- **Ersetze:** `.wt-theme-mein-neues-theme`

**Vorher:**
```css
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
```

**Nachher:**
```css
.wt-theme-mein-neues-theme .wt-genealogy-menu .dropdown-item.menu-list-burial {
```

Dies muss an **zwei Stellen** geÃ¤ndert werden:
1. MenÃ¼punkt-Styling (`.menu-list-burial`)
2. Icon-Styling (`.menu-list-burial::before`)

---

#### Schritt 4: Icon-Datei hinzufÃ¼gen (optional)

**Icon erstellen oder herunterladen:**
- Format: **SVG** (bevorzugt), PNG, oder JPG
- GrÃ¶ÃŸe: 24Ã—24 bis 64Ã—64 Pixel
- Hintergrund: Transparent (bei SVG/PNG)

**Icon ablegen:**
```
/modules_v4/burial-places-report/resources/icons/menu-list-burial-mein-neues-theme.svg
```

**CSS anpassen:**
```css
.wt-theme-mein-neues-theme .wt-genealogy-menu .dropdown-item.menu-list-burial::before {
    /* ... */
    background-image: url("../icons/menu-list-burial-mein-neues-theme.svg");
    /* ... */
}
```

---

#### Schritt 5: Layout anpassen (optional)

**Icon-GrÃ¶ÃŸe:**
```css
width: 1.50em;   /* GrÃ¶ÃŸer = 1.80em, Kleiner = 1.20em */
height: 1.50em;
```

**Icon-Position horizontal:**
```css
left: 0.35em;    /* Weiter links = 0.25em, Weiter rechts = 0.50em */
```

**Icon-Position vertikal:**
```css
top: 32%;        /* HÃ¶her = 25%, Tiefer = 40% */
```

**ZeilenhÃ¶he:**
```css
line-height: 1.90 !important;  /* Kompakter = 1.50, Luftiger = 2.20 */
```

**Padding (Abstand fÃ¼r Icon):**
```css
padding: 0rem 1rem 0.70rem 2.20em !important;
                            â†‘
                     Platz fÃ¼r Icon (2.0em - 2.5em)
```

---

#### Schritt 6: Testen

1. **Datei speichern**
2. **Browser-Cache leeren** (Strg+F5)
3. **Webtrees neu laden**
4. **Theme wechseln** zu "mein-neues-theme"
5. **MenÃ¼ Ã¶ffnen** â†’ "Listen" â†’ PrÃ¼fe Icon

**Debugging:**
```javascript
// Browser-Konsole (F12)
console.log(document.body.className);  // Theme prÃ¼fen

// Icon-URL testen
let img = new Image();
img.src = '/modules_v4/burial-places-report/resources/icons/menu-list-burial-mein-neues-theme.svg';
img.onload = () => console.log('âœ… Icon gefunden!');
img.onerror = () => console.log('âŒ Icon NICHT gefunden!');
```

---

## 12. PHP-Integration: headContent()

### 12.1 Ãœbersicht

Die Methode `headContent()` ist der zentrale Einstiegspunkt fÃ¼r das Icon-System. Sie wird von Webtrees automatisch aufgerufen und fÃ¼gt CSS in den `<head>`-Bereich ein.

**Interface-Voraussetzung:**
```php
class BurialPlacesReportModule extends AbstractModule implements 
    ModuleGlobalInterface  // â† Wichtig!
{
    use ModuleGlobalTrait;  // â† Wichtig!
    
    public function headContent(): string
    {
        // Wird automatisch von Webtrees aufgerufen
    }
}
```

### 12.2 Funktionsweise

**Ablauf:**
1. CSS-Dateien im `/resources/css/` Verzeichnis sammeln
2. CSS-Inhalte einlesen
3. Icon-Pfade korrigieren (relativ â†’ absolut)
4. CSS inline in `<style>`-Tag ausgeben

### 12.3 VollstÃ¤ndiger Code (siehe Anhang A)

**Kurzversion:**
```php
public function headContent(): string
{
    // 1. Pfade vorbereiten
    $css_dir_fs  = rtrim($this->resourcesFolder(), '/\\') . '/css/';
    $module_dir  = basename(dirname(__FILE__));
    $icons_base  = '/modules_v4/' . $module_dir . '/resources/icons/';

    // 2. CSS-Dateien sammeln
    $files = $this->collectCssFiles($css_dir_fs);

    // 3. CSS-Inhalte zusammenfassen
    $bundle = $this->bundleCssFiles($files, $css_dir_fs, $icons_base);

    // 4. Als <style>-Tag ausgeben
    if ($bundle !== '') {
        return "\n<style>\n" . $bundle . "\n</style>\n";
    }

    return '';
}
```

### 12.4 CSS-Dateien sammeln

**Algorithmus:**
```php
private function collectCssFiles(string $css_dir_fs): array
{
    $files = [];

    // 1. Universal-CSS hinzufÃ¼gen (falls vorhanden)
    if (is_file($css_dir_fs . 'menu-icons.universal.css')) {
        $files[] = 'menu-icons.universal.css';
    }

    // 2. Alle theme-spezifischen CSS-Dateien sammeln
    $all = glob($css_dir_fs . 'menu-icons.*.css');
    if (is_array($all)) {
        foreach ($all as $abs) {
            $bn = basename($abs);

            // AusschlieÃŸen: universal, none, template-*
            if ($bn === 'menu-icons.universal.css') {
                continue; // Bereits hinzugefÃ¼gt
            }
            if ($bn === 'menu-icons.none.css') {
                continue; // Spezielle Datei
            }
            if (str_starts_with($bn, 'menu-icons.template-')) {
                continue; // Template-Dateien
            }

            // Nur echte Theme-Dateien: menu-icons.<theme>.css
            if (preg_match('/^menu-icons\..+\.css$/', $bn)) {
                $files[] = $bn;
            }
        }
    }

    return $files;
}
```

**Ergebnis:**
```php
[
    'menu-icons.universal.css',
    'menu-icons.webtrees.css',
    'menu-icons.colors.css',
    'menu-icons.fab.css',
    // ...
]
```

### 12.5 CSS-Inhalte bÃ¼ndeln

**Pfad-Korrektur:**

CSS-Dateien verwenden relative Pfade:
```css
background-image: url("../icons/menu-list-burial-webtrees.svg");
```

Diese mÃ¼ssen zu absoluten Pfaden konvertiert werden:
```css
background-image: url("/modules_v4/burial-places-report/resources/icons/menu-list-burial-webtrees.svg");
```

**Code:**
```php
private function bundleCssFiles(array $files, string $css_dir_fs, string $icons_base): string
{
    $bundle = '';
    
    foreach ($files as $file) {
        $path = $css_dir_fs . $file;
        if (!is_file($path)) {
            continue;
        }

        $css = file_get_contents($path);
        if ($css === false) {
            continue;
        }

        // Icon-Pfade korrigieren
        $css = str_replace('../icons/webtrees/', $icons_base . 'webtrees/', $css);
        $css = str_replace('../icons/', $icons_base, $css);
        $css = str_replace('\\', '/', $css);  // Windows-Pfade normalisieren

        // CSS mit Kommentar hinzufÃ¼gen
        $bundle .= "\n/* {$file} */\n" . $css . "\n";
    }

    return $bundle;
}
```

### 12.6 HTML-Output

**Generiertes HTML:**
```html
<style>
/* menu-icons.universal.css */
.dropdown-item.menu-list-burial {
    /* ... */
    background-image: url("/modules_v4/burial-places-report/resources/icons/menu-list-burial-universal.svg");
}

/* menu-icons.webtrees.css */
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
    /* ... */
    background-image: url("/modules_v4/burial-places-report/resources/icons/menu-list-burial-webtrees.svg");
}

/* menu-icons.colors.css */
.wt-theme-colors .wt-genealogy-menu .dropdown-item.menu-list-burial {
    /* ... */
    background-image: url("/modules_v4/burial-places-report/resources/icons/menu-list-burial-colors.svg");
}
</style>
```

### 12.7 Performance-Optimierung

**Warum alle CSS-Dateien laden?**

Wir laden **alle** theme-spezifischen CSS-Dateien, nicht nur die des aktiven Themes.

**Vorteile:**
- âœ… Einfachere Implementierung (keine Theme-Erkennung nÃ¶tig)
- âœ… Theme-Wechsel ohne Reload funktioniert
- âœ… CSS-SpezifitÃ¤t regelt automatisch, welche Regel gilt
- âœ… Nur ~5-10 KB zusÃ¤tzliche Daten

**CSS-SpezifitÃ¤t:**
```css
/* Niedrige SpezifitÃ¤t (0,0,1,1) */
.dropdown-item.menu-list-burial {
    /* Universal-CSS */
}

/* HÃ¶here SpezifitÃ¤t (0,0,3,1) */
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
    /* Theme-spezifisches CSS Ã¼berschreibt Universal-CSS */
}
```

Das aktive Theme "gewinnt" automatisch durch hÃ¶here SpezifitÃ¤t!

---

## 13. CSS-Parameter im Detail

### 13.1 MenÃ¼punkt-Layout

```css
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
    /* â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
       POSITIONIERUNG
       â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â• */
    position: relative;
    /* âš ï¸ NICHT Ã„NDERN! Notwendig fÃ¼r absolute Positionierung des Icons */
    
    /* â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
       ZEILENHÃ–HE
       â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â• */
    line-height: 1.90 !important;
    /* ðŸ“ Anpassbar: 1.0 - 3.0
       â€¢ Niedrig (1.2-1.5) = Kompakt, wenig Abstand zwischen Zeilen
       â€¢ Mittel (1.6-2.0) = Standard, ausgewogener Abstand
       â€¢ Hoch (2.1-2.5) = Luftig, viel Abstand zwischen Zeilen
       
       Beispiele:
       line-height: 1.40 !important;  // Sehr kompakt
       line-height: 1.70 !important;  // Ausgewogen
       line-height: 2.20 !important;  // Sehr luftig
    */
    
    /* â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
       INNENABSTÃ„NDE (PADDING)
       â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â• */
    padding: 0rem 1rem 0.70rem 2.20em !important;
    /*       â†‘    â†‘    â†‘       â†‘
             TOP  RIGHT BOTTOM LEFT
    */
    
    /* ðŸ“ Anpassbar:
       â€¢ padding-top:    0rem - 0.8rem  (Abstand nach oben)
       â€¢ padding-right:  0.5rem - 1.5rem (Abstand nach rechts)
       â€¢ padding-bottom: 0.3rem - 1.0rem (Abstand nach unten)
       â€¢ padding-left:   2.0em - 2.8em  (Abstand nach links, Platz fÃ¼r Icon!)
       
       âš ï¸ WICHTIG: padding-left muss groÃŸ genug sein fÃ¼r das Icon!
       
       Empfohlene Werte fÃ¼r padding-left:
       â€¢ Kein Icon:        1.0em
       â€¢ Kleines Icon:     2.0em
       â€¢ Standard Icon:    2.2em - 2.5em
       â€¢ GroÃŸes Icon:      2.8em - 3.2em
       
       Beispiele:
       padding: 0.20rem 1rem 0.40rem 2.00em !important;  // Kompakt
       padding: 0.50rem 1rem 0.80rem 2.50em !important;  // Luftig
    */
}
```

### 13.2 Icon-Styling (::before)

```css
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial::before {
    /* â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
       PSEUDO-ELEMENT
       â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â• */
    content: "";
    /* âš ï¸ NICHT Ã„NDERN! Leerer String erzeugt Pseudo-Element */
    
    /* â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
       POSITIONIERUNG
       â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â• */
    position: absolute;
    /* âš ï¸ NICHT Ã„NDERN! Notwendig fÃ¼r freie Positionierung */
    
    /* â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
       HORIZONTALE POSITION
       â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â• */
    left: 0.35em;
    /* ðŸ“ Anpassbar: 0.2em - 0.8em
       â€¢ Kleiner Wert = Icon weiter links
       â€¢ GrÃ¶ÃŸer Wert = Icon weiter rechts
       
       Beispiele:
       left: 0.20em;  // Icon ganz links, nah am Rand
       left: 0.35em;  // Standard, etwas Abstand zum Rand
       left: 0.50em;  // Icon weiter rechts
       left: 0.65em;  // Icon noch weiter rechts
       
       ðŸ’¡ Tipp: Bei grÃ¶ÃŸeren Icons, left erhÃ¶hen!
    */
    
    /* â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
       VERTIKALE POSITION
       â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â• */
    top: 32%;
    /* ðŸ“ Anpassbar: 20% - 50%
       â€¢ Kleiner Wert = Icon hÃ¶her (nÃ¤her an Oberseite)
       â€¢ GrÃ¶ÃŸer Wert = Icon tiefer (nÃ¤her an Unterseite)
       
       Beispiele:
       top: 25%;  // Icon sehr hoch
       top: 32%;  // Standard (webtrees)
       top: 40%;  // Standard (universal)
       top: 45%;  // Icon sehr tief
       
       ðŸ’¡ Tipp: Bei groÃŸer line-height, top erhÃ¶hen!
    */
    
    transform: translateY(-50%);
    /* âš ï¸ NICHT Ã„NDERN! Zentriert Icon vertikal */
    
    /* â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
       ICON-GRÃ–SSE
       â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â• */
    width: 1.50em;
    /* ðŸ“ Anpassbar: 0.8em - 2.0em
       â€¢ 0.8em - 1.0em = Sehr klein
       â€¢ 1.0em - 1.3em = Klein
       â€¢ 1.3em - 1.7em = Standard
       â€¢ 1.7em - 2.0em = GroÃŸ
       
       Beispiele:
       width: 1.00em;  // Kleines Icon
       width: 1.50em;  // Standard Icon
       width: 1.80em;  // GroÃŸes Icon
    */
    
    height: 1.50em;
    /* ðŸ“ Anpassbar: 0.8em - 2.0em
       âš ï¸ WICHTIG: Sollte gleich wie width sein fÃ¼r quadratische Icons!
       
       ðŸ’¡ Tipp fÃ¼r nicht-quadratische Icons:
       width: 1.80em;
       height: 1.50em;
    */
    
    /* â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
       ICON-BILD
       â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â• */
    background-image: url("../icons/menu-list-burial-webtrees.svg");
    /* ðŸ“ Anpassbar: Pfad zum Icon-Bild
       
       Relative Pfade (werden in headContent() zu absoluten Pfaden konvertiert):
       ../icons/mein-icon.svg
       ../icons/webtrees/mein-icon.svg
       ../icons/custom/mein-icon.png
       
       UnterstÃ¼tzte Formate:
       â€¢ SVG (empfohlen) - Skaliert perfekt
       â€¢ PNG - Transparent, feste GrÃ¶ÃŸe
       â€¢ JPG - Nur bei farbigen Icons
       â€¢ WebP - Modern, kleine DateigrÃ¶ÃŸe
    */
    
    background-repeat: no-repeat;
    /* âš ï¸ NICHT Ã„NDERN! Icon soll nicht wiederholt werden */
    
    background-size: contain;
    /* âš ï¸ NICHT Ã„NDERN! Icon soll komplett sichtbar sein
       
       Alternativen:
       background-size: cover;    // Icon fÃ¼llt gesamten Bereich (ggf. abgeschnitten)
       background-size: 100% 100%;  // Icon wird gestreckt (verzerrt!)
    */
    
    background-position: center;
    /* âš ï¸ NICHT Ã„NDERN! Icon soll zentriert sein
       
       Alternativen:
       background-position: left center;   // Icon linksbÃ¼ndig
       background-position: right center;  // Icon rechtsbÃ¼ndig
    */
    
    pointer-events: none;
    /* âš ï¸ NICHT Ã„NDERN! Verhindert Klick-Probleme mit Icon */
}
```

### 13.3 Praxisbeispiele

#### Beispiel 1: GroÃŸes Icon, viel Platz

```css
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
    line-height: 2.20 !important;           /* Mehr ZeilenhÃ¶he */
    padding: 0.50rem 1rem 0.80rem 2.80em !important;  /* Mehr Platz fÃ¼r Icon */
}

.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial::before {
    left: 0.40em;        /* Etwas weiter rechts */
    top: 35%;            /* Etwas tiefer */
    width: 1.80em;       /* GrÃ¶ÃŸeres Icon */
    height: 1.80em;
}
```

#### Beispiel 2: Kleines Icon, kompakt

```css
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
    line-height: 1.40 !important;           /* Weniger ZeilenhÃ¶he */
    padding: 0.30rem 1rem 0.40rem 2.00em !important;  /* Weniger Platz */
}

.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial::before {
    left: 0.30em;        /* NÃ¤her am Rand */
    top: 30%;            /* HÃ¶her */
    width: 1.00em;       /* Kleineres Icon */
    height: 1.00em;
}
```

#### Beispiel 3: Kein Icon

```css
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
    line-height: 1.50 !important;
    padding: 0.40rem 1rem !important;  /* KEIN padding-left! */
}

.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial::before {
    content: none !important;  /* Kein Icon */
    display: none !important;
}
```

---

## 14. Icon-Dateien und Formate

### 14.1 UnterstÃ¼tzte Formate

| Format | Vorteile | Nachteile | Empfehlung |
|--------|----------|-----------|------------|
| **SVG** | âœ… Skaliert perfekt<br>âœ… Kleine DateigrÃ¶ÃŸe<br>âœ… Editierbar<br>âœ… Transparent | âŒ Komplexe Icons kÃ¶nnen groÃŸ werden | â­ **Empfohlen** |
| **PNG** | âœ… Transparent<br>âœ… Breite UnterstÃ¼tzung | âŒ Feste GrÃ¶ÃŸe<br>âŒ Pixelig bei Skalierung | âœ… Alternative |
| **JPG** | âœ… Kleine DateigrÃ¶ÃŸe<br>âœ… Gute Farbwiedergabe | âŒ Kein Transparenz<br>âŒ Verlustbehaftet | âš ï¸ Nur fÃ¼r Fotos |
| **WebP** | âœ… Sehr kleine DateigrÃ¶ÃŸe<br>âœ… Transparent | âŒ Neuere Browser nÃ¶tig | âœ… Modern |

### 14.2 Icon-Spezifikationen

**GrÃ¶ÃŸe:**
- **Minimum:** 24Ã—24 Pixel
- **Empfohlen:** 48Ã—48 oder 64Ã—64 Pixel
- **Maximum:** 128Ã—128 Pixel (grÃ¶ÃŸer ist unnÃ¶tig)

**Format:**
- **Quadratisch** (gleiche Breite und HÃ¶he)
- Bei nicht-quadratischen Icons: CSS `width`/`height` anpassen

**Hintergrund:**
- **SVG/PNG:** Transparent
- **JPG:** Passende Hintergrundfarbe

**Farben:**
- **Einfarbig:** Am besten fÃ¼r Icons
- **Mehrfarbig:** MÃ¶glich, aber sparsam einsetzen
- **Farbe anpassen:** Icons sollten zum Theme passen

### 14.3 Icon-Erstellung

#### SVG erstellen (Inkscape)

1. **Inkscape Ã¶ffnen**
2. **Neues Dokument:** 48Ã—48 px
3. **Icon zeichnen** (einfache Formen)
4. **Speichern als:** `menu-list-burial-mein-theme.svg`
5. **Optimieren:**
   ```bash
   # SVGO (SVG Optimizer)
   svgo menu-list-burial-mein-theme.svg
   ```

#### PNG erstellen (GIMP)

1. **GIMP Ã¶ffnen**
2. **Neues Bild:** 64Ã—64 px
3. **Hintergrund:** Transparent
4. **Icon zeichnen**
5. **Exportieren als PNG**
6. **Optimieren:**
   ```bash
   # PNGCrush (PNG Optimizer)
   pngcrush -brute menu-list-burial-mein-theme.png menu-list-burial-mein-theme-optimized.png
   ```

### 14.4 Icon-Bibliotheken

**Kostenlose Icon-Bibliotheken:**
- [Font Awesome](https://fontawesome.com/) (als SVG exportieren)
- [Material Icons](https://fonts.google.com/icons)
- [Heroicons](https://heroicons.com/)
- [Feather Icons](https://feathericons.com/)
- [Bootstrap Icons](https://icons.getbootstrap.com/)

**Lizenz beachten:** PrÃ¼fe immer die Lizenzbedingungen!

### 14.5 Icon-Namenskonvention

**Schema:**
```
menu-list-burial-<theme-name>.<format>
```

**Beispiele:**
```
menu-list-burial-webtrees.svg
menu-list-burial-colors.svg
menu-list-burial-fab.png
menu-list-burial-minimal.svg
menu-list-burial-universal.svg
```

**Wichtig:** Theme-Name muss mit CSS-Datei Ã¼bereinstimmen!

```
menu-icons.webtrees.css     â†”  menu-list-burial-webtrees.svg
menu-icons.colors.css       â†”  menu-list-burial-colors.svg
```

---

## 15. Fallback-Mechanismus

### 15.1 Funktionsweise

**Hierarchie:**
```
1. Theme-spezifisches CSS (hÃ¶chste PrioritÃ¤t)
   â””â”€ .wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial
   
2. Universal-CSS (niedrigste PrioritÃ¤t)
   â””â”€ .dropdown-item.menu-list-burial
```

**CSS-SpezifitÃ¤t:**
```css
/* SpezifitÃ¤t: 0,0,3,1 (Theme-Klasse + 2 weitere Klassen + Element) */
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
    background-image: url("webtrees.svg");  /* â† Wird verwendet! */
}

/* SpezifitÃ¤t: 0,0,2,0 (2 Klassen) */
.dropdown-item.menu-list-burial {
    background-image: url("universal.svg");  /* â† Wird Ã¼berschrieben */
}
```

Das theme-spezifische CSS "gewinnt" durch hÃ¶here SpezifitÃ¤t!

### 15.2 Ablauf

**Szenario 1: Theme mit eigener CSS-Datei**
```
Theme: webtrees
â”œâ”€ menu-icons.webtrees.css existiert
â””â”€ Verwendet: menu-icons.webtrees.css (Icon: webtrees.svg)
```

**Szenario 2: Theme ohne eigene CSS-Datei**
```
Theme: mein-neues-theme
â”œâ”€ menu-icons.mein-neues-theme.css existiert NICHT
â””â”€ Fallback: menu-icons.universal.css (Icon: universal.svg)
```

**Szenario 3: Beide CSS-Dateien vorhanden**
```
Theme: webtrees
â”œâ”€ menu-icons.universal.css geladen (niedrige SpezifitÃ¤t)
â”œâ”€ menu-icons.webtrees.css geladen (hohe SpezifitÃ¤t)
â””â”€ Verwendet: menu-icons.webtrees.css (Ã¼berschreibt universal)
```

### 15.3 Warum beide laden?

**Vorteile:**
- âœ… Einfachere Implementierung (keine komplexe Logik)
- âœ… Theme-Wechsel ohne Reload funktioniert
- âœ… CSS-Browser-Cache wird optimal genutzt
- âœ… Nur ~10 KB zusÃ¤tzliche Daten

**Nachteil:**
- âŒ Etwas mehr Daten (~5-10 KB)

**Entscheidung:** Vorteile Ã¼berwiegen!

### 15.4 Code-Beispiel

```php
public function headContent(): string
{
    $css_dir_fs = rtrim($this->resourcesFolder(), '/\\') . '/css/';
    $files = [];

    // 1. Universal-CSS IMMER laden (Fallback)
    if (is_file($css_dir_fs . 'menu-icons.universal.css')) {
        $files[] = 'menu-icons.universal.css';
    }

    // 2. ALLE theme-spezifischen CSS laden
    $all = glob($css_dir_fs . 'menu-icons.*.css');
    if (is_array($all)) {
        foreach ($all as $abs) {
            $bn = basename($abs);
            
            // Universal bereits hinzugefÃ¼gt
            if ($bn === 'menu-icons.universal.css') {
                continue;
            }
            
            // Nur echte Theme-Dateien
            if (preg_match('/^menu-icons\..+\.css$/', $bn)) {
                $files[] = $bn;
            }
        }
    }

    // 3. CSS zusammenfassen und ausgeben
    return $this->bundleAndOutputCss($files);
}
```

**Reihenfolge im HTML:**
```html
<style>
/* 1. Universal (niedrige SpezifitÃ¤t) */
.dropdown-item.menu-list-burial { ... }

/* 2. Theme-spezifisch (hohe SpezifitÃ¤t) */
.wt-theme-webtrees .dropdown-item.menu-list-burial { ... }
.wt-theme-colors .dropdown-item.menu-list-burial { ... }
.wt-theme-fab .dropdown-item.menu-list-burial { ... }
</style>
```

Das aktive Theme Ã¼berschreibt automatisch das Universal-CSS!

---

# Teil 4: Datenverarbeitung

## 16. Datenfluss und Verarbeitung

### 16.1 Request-Lifecycle

```
1. USER REQUEST
   â”‚
   â”œâ”€ GET /tree/demo/burial-places
   â”‚  â””â”€â†’ handle() Methode
   â”‚
   â”œâ”€ POST /tree/demo/burial-places (mit Filtern)
   â”‚  â””â”€â†’ handle() Methode
   â”‚
   â””â”€ POST /tree/demo/burial-places/export (CSV)
      â””â”€â†’ exportCsvAction() Methode
      
2. CONTROLLER (module.php)
   â”‚
   â”œâ”€ Request validieren
   â”œâ”€ Parameter extrahieren
   â”œâ”€ Tree-Objekt holen
   â””â”€â†’ Daten verarbeiten
   
3. DATA PROCESSING
   â”‚
   â”œâ”€ Individuals aus Tree laden
   â”œâ”€ BURI-Facts extrahieren
   â”œâ”€ Friedhofsinformationen ermitteln
   â”œâ”€ Daten gruppieren (Ort + Institution)
   â””â”€â†’ Daten an View Ã¼bergeben
   
4. VIEW RENDERING
   â”‚
   â”œâ”€ Template laden (list-page.phtml)
   â”œâ”€ Daten einsetzen
   â””â”€â†’ HTML generieren
   
5. RESPONSE
   â””â”€â†’ HTML an Browser senden
```

### 16.2 handle() - Hauptmethode

```php
public function handle(ServerRequestInterface $request): ResponseInterface
{
    // 1. Request-Daten extrahieren
    $tree = Validator::attributes($request)->tree();
    $user = Validator::attributes($request)->user();
    
    // 2. Berechtigungen prÃ¼fen
    if (!Auth::isManager($tree, $user)) {
        return redirect(route('tree-page', ['tree' => $tree->name()]));
    }
    
    // 3. POST-Parameter (Filter)
    $filter_place = Validator::parsedBody($request)->string('filter_place', '');
    $filter_institution = Validator::parsedBody($request)->string('filter_institution', '');
    $filter_info = Validator::parsedBody($request)->string('filter_info', 'all');
    $sort_by = Validator::parsedBody($request)->string('sort_by', 'place');
    
    // 4. CSV-Export?
    if (Validator::parsedBody($request)->string('export_csv', '') !== '') {
        return $this->exportCsvAction($request);
    }
    
    // 5. Einstellungen laden
    $fallback_order = $this->getTreePreference($tree, 'fallback_order', self::DEFAULT_FALLBACK_ORDER);
    $plac_keywords = $this->getTreePreference($tree, 'plac_keywords', self::DEFAULT_PLAC_KEYWORDS);
    $diagnosis_mode = $this->getTreePreference($tree, 'diagnosis_mode', self::DEFAULT_DIAGNOSIS_MODE) === '1';
    
    // 6. Daten sammeln
    $institutions = $this->collectInstitutions(
        $tree,
        $filter_place,
        $filter_institution,
        $filter_info,
        $sort_by,
        $fallback_order,
        $plac_keywords,
        $diagnosis_mode
    );
    
    // 7. View rendern
    return $this->viewResponse($this->name() . '::list-page', [
        'title' => $this->title(),
        'tree' => $tree,
        'institutions' => $institutions,
        'filter_place' => $filter_place,
        'filter_institution' => $filter_institution,
        'filter_info' => $filter_info,
        'sort_by' => $sort_by,
        'diagnosis_mode' => $diagnosis_mode,
    ]);
}
```

### 16.3 collectInstitutions() - Datensammlung

```php
/**
 * Sammelt Bestattungen gruppiert nach Ort und Institution
 */
private function collectInstitutions(
    Tree $tree,
    string $filter_place,
    string $filter_institution,
    string $filter_info,
    string $sort_by,
    string $fallback_order,
    string $plac_keywords,
    bool $diagnosis_mode
): array {
    $institutions = [];
    $sources = explode(',', $fallback_order);
    $keywords = explode(',', $plac_keywords);
    
    // 1. Alle Personen durchgehen
    foreach ($tree->individuals() as $individual) {
        
        // 2. Bestattungen suchen
        $burials = $individual->facts(['BURI']);
        
        foreach ($burials as $burial) {
            
            // 3. Bestattungsort ermitteln
            $place = $burial->place()->gedcomName();
            
            // 4. Friedhof/Institution ermitteln
            $result = $this->determineBurialPlace(
                $burial,
                $tree,
                $sources,
                $keywords
            );
            
            $institution = $result['value'];
            $source = $result['source'];
            
            // 5. Filter anwenden
            if (!$this->matchesFilters($place, $institution, $source, $filter_place, $filter_institution, $filter_info)) {
                continue;
            }
            
            // 6. GruppierungsschlÃ¼ssel erstellen
            $key = $place . '|' . $institution;
            
            // 7. Zur Gruppe hinzufÃ¼gen
            if (!isset($institutions[$key])) {
                $institutions[$key] = [
                    'place' => $place,
                    'institution' => $institution,
                    'source' => $source,
                    'individuals' => [],
                ];
            }
            
            $institutions[$key]['individuals'][] = [
                'individual' => $individual,
                'burial' => $burial,
                'burial_date' => $burial->date()->display(),
                'birth_date' => $this->getBirthDate($individual),
                'death_date' => $this->getDeathDate($individual),
                'age' => $this->calculateAge($individual),
            ];
        }
    }
    
    // 8. Sortieren
    if ($sort_by === 'institution') {
        uasort($institutions, function($a, $b) {
            return $a['institution'] <=> $b['institution'];
        });
    } else {
        uasort($institutions, function($a, $b) {
            return $a['place'] <=> $b['place'];
        });
    }
    
    return array_values($institutions);
}
```

---

## 17. GEDCOM-Verarbeitung

### 17.1 GEDCOM-Struktur

**Bestattungs-Ereignis:**
```gedcom
0 @I1@ INDI
1 NAME Max /Mustermann/
1 BURI
2 DATE 15 MAR 1950
2 PLAC Wien, Ã–sterreich
2 AGNC Zentralfriedhof Wien
2 ADDR Simmeringer HauptstraÃŸe 234
3 CITY Wien
3 POST 1110
2 NOTE Grab 12A, Gruppe 15
```

### 17.2 determineBurialPlace() - Friedhof ermitteln

```php
/**
 * Ermittelt Friedhof/Institution aus GEDCOM-Daten
 * 
 * @param Fact  $burial        BURI-Fact
 * @param Tree  $tree          Stammbaum
 * @param array $sources       ['AGNC', 'PLAC', 'NOTE']
 * @param array $keywords      ['Friedhof', 'Cemetery', ...]
 * @return array ['value' => string, 'source' => string]
 */
private function determineBurialPlace(
    Fact $burial,
    Tree $tree,
    array $sources,
    array $keywords
): array {
    foreach ($sources as $source) {
        switch ($source) {
            case 'AGNC':
                $value = $this->extractAGNC($burial);
                if ($value !== '') {
                    return ['value' => $value, 'source' => 'AGNC'];
                }
                break;
                
            case 'NOTE':
                $value = $this->extractNOTE($burial, $keywords);
                if ($value !== '') {
                    return ['value' => $value, 'source' => 'NOTE'];
                }
                break;
                
            case 'PLAC':
                $value = $this->extractPLAC($burial, $keywords);
                if ($value !== '') {
                    return ['value' => $value, 'source' => 'PLAC'];
                }
                break;
        }
    }
    
    // Nichts gefunden
    return ['value' => I18N::translate('Unknown'), 'source' => 'NONE'];
}
```

### 17.3 AGNC extrahieren

```php
/**
 * Extrahiert AGNC-Feld (Institution)
 */
private function extractAGNC(Fact $burial): string
{
    // Zugriff auf GEDCOM-Rohtext
    $gedcom = $burial->gedcom();
    
    // Pattern: 2 AGNC <value>
    if (preg_match('/\n2 AGNC (.+)/', $gedcom, $matches)) {
        return trim($matches[1]);
    }
    
    return '';
}
```

**Beispiel:**
```gedcom
1 BURI
2 AGNC Zentralfriedhof Wien
```
â†’ Output: `"Zentralfriedhof Wien"`

### 17.4 NOTE extrahieren (mit Keywords)

```php
/**
 * Extrahiert erste Zeile aus NOTE, wenn Keyword enthalten
 */
private function extractNOTE(Fact $burial, array $keywords): string
{
    $gedcom = $burial->gedcom();
    
    // Pattern: 2 NOTE <value>
    if (preg_match('/\n2 NOTE (.+?)(?:\n|$)/s', $gedcom, $matches)) {
        $note_first_line = trim(explode("\n", $matches[1])[0]);
        
        // PrÃ¼fe auf Keywords
        foreach ($keywords as $keyword) {
            if (stripos($note_first_line, $keyword) !== false) {
                return $note_first_line;
            }
        }
    }
    
    return '';
}
```

**Beispiel:**
```gedcom
1 BURI
2 NOTE Friedhof St. Marx, Gruppe 15
3 CONT Weitere Informationen...
```
â†’ Erste Zeile: `"Friedhof St. Marx, Gruppe 15"`  
â†’ Keyword `"Friedhof"` gefunden  
â†’ Output: `"Friedhof St. Marx, Gruppe 15"`

### 17.5 PLAC extrahieren (mit Keywords)

```php
/**
 * Extrahiert PLAC-Feld, wenn Keyword enthalten
 */
private function extractPLAC(Fact $burial, array $keywords): string
{
    $place = $burial->place()->gedcomName();
    
    // PrÃ¼fe auf Keywords
    foreach ($keywords as $keyword) {
        if (stripos($place, $keyword) !== false) {
            return $place;
        }
    }
    
    return '';
}
```

**Beispiel:**
```gedcom
1 BURI
2 PLAC Wien, Zentralfriedhof
```
â†’ PLAC: `"Wien, Zentralfriedhof"`  
â†’ Keyword `"Zentralfriedhof"` gefunden  
â†’ Output: `"Wien, Zentralfriedhof"`

### 17.6 Fallback-Logik Beispiel

**Konfiguration:**
```
Fallback-Reihenfolge: AGNC, NOTE, PLAC
Keywords: Friedhof, Cemetery, CimetiÃ¨re
```

**GEDCOM-Daten:**
```gedcom
1 BURI
2 DATE 15 MAR 1950
2 PLAC Wien, Zentralfriedhof
2 NOTE Bestattung im Familiengrab
```

**Ablauf:**
1. **AGNC prÃ¼fen:** Nicht vorhanden â†’ weiter
2. **NOTE prÃ¼fen:** `"Bestattung im Familiengrab"` â†’ Kein Keyword â†’ weiter
3. **PLAC prÃ¼fen:** `"Wien, Zentralfriedhof"` â†’ Keyword `"Zentralfriedhof"` gefunden! â†’ **Treffer!**

**Ergebnis:**
```php
[
    'value' => 'Wien, Zentralfriedhof',
    'source' => 'PLAC'
]
```

---

## 18. Datenbank-Interaktion

### 18.1 Preferences speichern

```php
/**
 * Speichert baum-spezifische Einstellung
 */
private function setTreePreference(Tree $tree, string $key, string $value): void
{
    $full_key = 'burial_places_' . $key;
    $tree->setPreference($full_key, $value);
}
```

**Datenbank-Query (intern von Webtrees):**
```sql
INSERT INTO `wt_tree_preference` (`tree_id`, `setting_name`, `setting_value`)
VALUES (1, 'burial_places_fallback_order', 'AGNC,NOTE,PLAC')
ON DUPLICATE KEY UPDATE `setting_value` = 'AGNC,NOTE,PLAC';
```

### 18.2 Preferences laden

```php
/**
 * LÃ¤dt baum-spezifische Einstellung
 */
private function getTreePreference(Tree $tree, string $key, string $default): string
{
    $full_key = 'burial_places_' . $key;
    return $tree->getPreference($full_key, $default);
}
```

**Datenbank-Query (intern von Webtrees):**
```sql
SELECT `setting_value`
FROM `wt_tree_preference`
WHERE `tree_id` = 1
  AND `setting_name` = 'burial_places_fallback_order';
```

### 18.3 Gespeicherte Preferences

| Key | Typ | Default | Beschreibung |
|-----|-----|---------|--------------|
| `burial_places_fallback_order` | string | `'AGNC,PLAC,NOTE'` | Reihenfolge der Feldsuche |
| `burial_places_plac_keywords` | string | `'Friedhof,...'` | Keyword-Liste |
| `burial_places_diagnosis_mode` | string | `'0'` | Diagnosemodus (0/1) |
| `burial_places_show_agnc_always` | string | `'0'` | AGNC immer zeigen (0/1) |

### 18.4 Individuals laden

```php
// Alle Personen aus Tree
$individuals = $tree->individuals();

// Mit Query Builder (fÃ¼r groÃŸe Datenmengen)
$individuals = DB::table('individuals')
    ->where('i_file', '=', $tree->id())
    ->get();
```

### 18.5 Facts laden

```php
// Alle BURI-Facts einer Person
$burials = $individual->facts(['BURI']);

// Alle Facts (inkl. BURI)
$all_facts = $individual->facts();

// Erster BURI-Fact
$burial = $individual->facts(['BURI'])->first();
```

---

## 19. CSV-Export-FunktionalitÃ¤t

### 19.1 exportCsvAction()

```php
public function exportCsvAction(ServerRequestInterface $request): ResponseInterface
{
    // 1. Daten sammeln (gleiche Logik wie handle())
    $tree = Validator::attributes($request)->tree();
    $institutions = $this->collectInstitutions(...);
    
    // 2. CSV-Daten erstellen
    $csv_data = $this->generateCsvData($institutions);
    
    // 3. Dateiname generieren
    $filename = 'burial-places-' . $tree->name() . '-' . date('Y-m-d') . '.csv';
    
    // 4. Response mit CSV
    return response($csv_data)
        ->withHeader('Content-Type', 'text/csv; charset=UTF-8')
        ->withHeader('Content-Disposition', 'attachment; filename="' . $filename . '"');
}
```

### 19.2 generateCsvData()

```php
/**
 * Generiert CSV-Daten mit UTF-8 BOM
 */
private function generateCsvData(array $institutions): string
{
    // 1. Output-Buffer starten
    ob_start();
    
    // 2. UTF-8 BOM fÃ¼r Excel
    echo "\xEF\xBB\xBF";
    
    // 3. CSV-Handle Ã¶ffnen
    $fp = fopen('php://output', 'w');
    
    // 4. Header-Zeile
    fputcsv($fp, [
        I18N::translate('Burial place'),
        I18N::translate('Cemetery/Institution'),
        I18N::translate('Name'),
        I18N::translate('Death date'),
        I18N::translate('Burial date'),
        I18N::translate('Address'),
        I18N::translate('Note'),
        I18N::translate('URL'),
    ], ';');
    
    // 5. Datenzeilen
    foreach ($institutions as $data) {
        foreach ($data['individuals'] as $entry) {
            fputcsv($fp, [
                $data['place'],
                $data['institution'],
                $entry['individual']->fullName(),
                $entry['death_date'],
                $entry['burial_date'],
                $this->getAddress($entry['burial']),
                $this->getNote($entry['burial']),
                $entry['individual']->url(),
            ], ';');
        }
    }
    
    // 6. Handle schlieÃŸen
    fclose($fp);
    
    // 7. Buffer holen und zurÃ¼ckgeben
    return ob_get_clean();
}
```

### 19.3 CSV-Format

**Trennzeichen:** Semikolon (`;`)  
**Textbegrenzung:** AnfÃ¼hrungszeichen (`"`)  
**Encoding:** UTF-8 mit BOM  
**Zeilenumbruch:** `\n`

**Beispiel-Output:**
```csv
"Bestattungsort";"Friedhof/Institution";"Name";"Sterbedatum";"Bestattungsdatum";"Adresse";"Notiz";"URL"
"Wien, Ã–sterreich";"Zentralfriedhof Wien";"Max Mustermann";"12 MÃ¤r 1950";"15 MÃ¤r 1950";"Simmeringer HauptstraÃŸe 234";"Grab 12A, Gruppe 15";"https://example.com/individual/I1"
"Berlin, Deutschland";"Friedhof St. Hedwig";"Anna Schmidt";"5 Jun 1965";"8 Jun 1965";"Liebenwalder StraÃŸe 10";"Familiengrab";"https://example.com/individual/I2"
```

### 19.4 UTF-8 BOM

**Warum UTF-8 BOM?**
- Excel erkennt UTF-8 nur mit BOM
- Ohne BOM werden Umlaute falsch dargestellt

**BOM-Bytes:**
```php
echo "\xEF\xBB\xBF";  // UTF-8 BOM
```

---

# Teil 5: Technische Details

## 20. Sicherheit und Validierung

### 20.1 CSRF-Schutz

**Problem:** Cross-Site Request Forgery (CSRF) Angriffe

**LÃ¶sung:** CSRF-Token in jedem Formular

```php
// In View-Template
<form method="post">
    <?= csrf_field() ?>
    <!-- Formular-Felder -->
</form>
```

**Generiertes HTML:**
```html
<form method="post">
    <input type="hidden" name="_csrf" value="abc123...xyz">
    <!-- Formular-Felder -->
</form>
```

**Validierung (automatisch von Webtrees):**
```php
// Webtrees prÃ¼ft automatisch CSRF-Token
// Fehler â†’ HTTP 403 Forbidden
```

### 20.2 Input-Validierung

```php
use Fisharebest\Webtrees\Validator;

// String mit Default-Wert
$filter_place = Validator::parsedBody($request)->string('filter_place', '');

// Integer
$limit = Validator::parsedBody($request)->integer('limit', 100);

// Boolean
$enabled = Validator::parsedBody($request)->boolean('enabled', false);

// Enum (bestimmte Werte)
$sort_by = Validator::parsedBody($request)->isInArray(['place', 'institution'])
                                          ->string('sort_by');
```

**Vorteile:**
- âœ… Typ-sichere Extraktion
- âœ… Default-Werte
- âœ… Keine ungÃ¼ltigen Daten

### 20.3 XSS-Schutz

**Problem:** Cross-Site Scripting (XSS) Angriffe

**LÃ¶sung:** Output-Escaping

```php
// In View-Templates: IMMER e() verwenden!

// âœ… Richtig
<?= e($user_input) ?>

// âŒ FALSCH - XSS-AnfÃ¤llig!
<?= $user_input ?>

// âœ… Richtig fÃ¼r HTML-Attribute
<input value="<?= e($value) ?>">

// âœ… Richtig fÃ¼r URLs
<a href="<?= e($url) ?>">Link</a>
```

**Beispiel XSS-Angriff:**
```php
$user_input = '<script>alert("XSS")</script>';

// âœ… Mit e()
echo e($user_input);
// Output: &lt;script&gt;alert(&quot;XSS&quot;)&lt;/script&gt;

// âŒ Ohne e()
echo $user_input;
// Output: <script>alert("XSS")</script>  â† Wird ausgefÃ¼hrt!
```

### 20.4 SQL-Injection Schutz

**Problem:** SQL-Injection Angriffe

**LÃ¶sung:** Prepared Statements (automatisch von Webtrees/Laravel)

```php
// âœ… Richtig - Prepared Statement
DB::table('individuals')
  ->where('i_id', '=', $user_input)
  ->get();

// âŒ FALSCH - SQL-Injection anfÃ¤llig!
DB::select("SELECT * FROM individuals WHERE i_id = '$user_input'");
```

### 20.5 BerechtigungsprÃ¼fung

```php
use Fisharebest\Webtrees\Auth;

// Ist Benutzer Admin?
if (Auth::isAdmin($user)) {
    // Zugriff erlaubt
}

// Ist Benutzer Manager fÃ¼r diesen Baum?
if (Auth::isManager($tree, $user)) {
    // Zugriff erlaubt
}

// Ist Benutzer Mitglied?
if (Auth::isMember($tree, $user)) {
    // Zugriff erlaubt
}

// VollstÃ¤ndige PrÃ¼fung
public function handle(ServerRequestInterface $request): ResponseInterface
{
    $tree = Validator::attributes($request)->tree();
    $user = Validator::attributes($request)->user();
    
    if (!Auth::isManager($tree, $user)) {
        return redirect(route('tree-page', ['tree' => $tree->name()]));
    }
    
    // ... Rest der Methode
}
```

---

## 21. Performance-Optimierung

### 21.1 Caching-Strategie

**Problem:** Datensammlung ist rechenintensiv bei groÃŸen StammbÃ¤umen

**LÃ¶sung:** Ergebnisse cachen

```php
use Illuminate\Support\Facades\Cache;

public function handle(ServerRequestInterface $request): ResponseInterface
{
    $tree = Validator::attributes($request)->tree();
    
    // Cache-Key erstellen
    $cache_key = 'burial-places-' . $tree->id() . '-' . md5($filter_place . $filter_institution);
    
    // Aus Cache laden oder berechnen
    $institutions = Cache::remember($cache_key, 3600, function() use ($tree, $filter_place, $filter_institution) {
        return $this->collectInstitutions($tree, $filter_place, $filter_institution, ...);
    });
    
    // ... Rest
}
```

**Cache-Invalidierung:**
```php
// Bei Ã„nderungen am Baum: Cache lÃ¶schen
Cache::forget('burial-places-' . $tree->id() . '-*');
```

### 21.2 Lazy Loading

**Problem:** Alle Daten werden sofort geladen, auch wenn nicht benÃ¶tigt

**LÃ¶sung:** Daten erst bei Bedarf laden

```php
// âŒ Ineffizient
foreach ($tree->individuals() as $individual) {
    $burials = $individual->facts(['BURI']);
    foreach ($burials as $burial) {
        // ...
    }
}

// âœ… Effizienter mit Query Builder
$burials = DB::table('individuals')
    ->join('facts', 'facts.f_i_id', '=', 'individuals.i_id')
    ->where('individuals.i_file', '=', $tree->id())
    ->where('facts.f_type', '=', 'BURI')
    ->select('individuals.*', 'facts.*')
    ->get();
```

### 21.3 Pagination

**Problem:** Zu viele Ergebnisse auf einer Seite

**LÃ¶sung:** Pagination implementieren

```php
use Illuminate\Pagination\Paginator;

public function handle(ServerRequestInterface $request): ResponseInterface
{
    $page = Validator::queryParams($request)->integer('page', 1);
    $per_page = 50;
    
    // Alle Daten sammeln
    $all_institutions = $this->collectInstitutions(...);
    
    // Pagination
    $total = count($all_institutions);
    $paginator = new Paginator(
        array_slice($all_institutions, ($page - 1) * $per_page, $per_page),
        $per_page,
        $page,
        ['path' => $request->getUri()->getPath()]
    );
    
    return $this->viewResponse($this->name() . '::list-page', [
        'institutions' => $paginator->items(),
        'paginator' => $paginator,
    ]);
}
```

### 21.4 CSS-Minimierung

**headContent() Output reduzieren:**

```php
public function headContent(): string
{
    // ... CSS sammeln ...
    
    // Kommentare und Whitespace entfernen (in Produktion)
    if (!defined('WT_DEBUG') || !WT_DEBUG) {
        $bundle = preg_replace('/\/\*.*?\*\//s', '', $bundle);  // Kommentare
        $bundle = preg_replace('/\s+/', ' ', $bundle);          // Whitespace
        $bundle = trim($bundle);
    }
    
    return "\n<style>\n" . $bundle . "\n</style>\n";
}
```

**Vorher (30 KB):**
```css
/* ============================================================
   webtrees â€“ MenÃ¼punkt "Bestattungsorte"
   ============================================================ */

.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
    position: relative;
    line-height: 1.90 !important;
    /* ... viele Kommentare ... */
}
```

**Nachher (18 KB):**
```css
.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial{position:relative;line-height:1.90 !important;}
```

---

## 22. Error Handling

### 22.1 Try-Catch BlÃ¶cke

```php
public function handle(ServerRequestInterface $request): ResponseInterface
{
    try {
        $tree = Validator::attributes($request)->tree();
        $institutions = $this->collectInstitutions($tree, ...);
        
        return $this->viewResponse(...);
        
    } catch (\Exception $e) {
        // Fehler loggen
        Log::error('Burial Places Report Error: ' . $e->getMessage());
        
        // Benutzerfreundliche Fehlermeldung
        FlashMessages::addMessage(
            I18N::translate('An error occurred. Please try again.'),
            'danger'
        );
        
        // ZurÃ¼ck zur Startseite
        return redirect(route('tree-page', ['tree' => $tree->name()]));
    }
}
```

### 22.2 Validierungsfehler

```php
public function postAdminAction(ServerRequestInterface $request): ResponseInterface
{
    $tree = Validator::attributes($request)->tree();
    $plac_keywords_raw = Validator::parsedBody($request)->string('plac_keywords', '');
    
    // Validierung
    $validation_result = $this->validateKeywords($plac_keywords_raw);
    
    if (!$validation_result['valid']) {
        // Fehlermeldung anzeigen
        FlashMessages::addMessage($validation_result['error'], 'danger');
        
        // ZurÃ¼ck zum Formular (mit Eingaben)
        return redirect($this->getConfigLink());
    }
    
    // Speichern
    $this->setTreePreference($tree, 'plac_keywords', $validation_result['keywords']);
    
    // Erfolgsmeldung
    FlashMessages::addMessage(I18N::translate('Settings saved'), 'success');
    
    return redirect($this->getConfigLink());
}
```

### 22.3 Logging

```php
use Psr\Log\LoggerInterface;
use Fisharebest\Webtrees\Registry;

// Logger aus Container holen
$logger = Registry::container()->get(LoggerInterface::class);

// Error
$logger->error('Burial Places: Failed to load CSS file', [
    'file' => $css_file,
    'error' => $exception->getMessage(),
]);

// Warning
$logger->warning('Burial Places: No AGNC field found', [
    'individual' => $individual->xref(),
]);

// Info
$logger->info('Burial Places: Settings updated', [
    'tree' => $tree->name(),
    'user' => $user->email(),
]);
```

### 22.4 Graceful Degradation

**Beispiel: Icon-Loading**

```php
public function headContent(): string
{
    $css_dir_fs = rtrim($this->resourcesFolder(), '/\\') . '/css/';
    
    // PrÃ¼fe ob CSS-Verzeichnis existiert
    if (!is_dir($css_dir_fs)) {
        // Fehler loggen, aber nicht abbrechen
        Log::warning('Burial Places: CSS directory not found', ['path' => $css_dir_fs]);
        return '';  // Kein CSS, aber Modul funktioniert weiter
    }
    
    // CSS-Dateien sammeln
    $files = glob($css_dir_fs . 'menu-icons.*.css');
    
    if (!is_array($files) || count($files) === 0) {
        // Keine CSS-Dateien gefunden
        Log::info('Burial Places: No CSS files found');
        return '';  // Modul funktioniert ohne Icons
    }
    
    // ... Rest
}
```

---

## 23. Troubleshooting Guide

### 23.1 HÃ¤ufige Probleme

#### Problem 1: Icon wird nicht angezeigt

**Symptome:**
- MenÃ¼punkt ohne Icon
- Universal-Icon statt theme-spezifischem Icon

**Diagnose:**

**Methode 1: Modul-Debug-Kommentar prÃ¼fen (SCHNELLSTE METHODE!) â­**

1. Rechtsklick auf Seite â†’ "Seitenquelltext anzeigen"
2. Suche nach `Detected Theme:` (Strg+F)
3. PrÃ¼fe die Ausgabe:

```html
<!-- Burial Places Module - Icon CSS -->
<!-- Detected Theme: _jc-theme-webtrees_ -->
<!-- Loaded CSS files: menu-icons_webtrees_.css -->
<style>
... (CSS-Code) ...
</style>
```

**Interpretation:**
- âœ… Theme erkannt? â†’ `Detected Theme: _jc-theme-webtrees_`
- âœ… CSS-Datei geladen? â†’ `Loaded CSS files: menu-icons_webtrees_.css`
- âœ… CSS-Code vorhanden? â†’ `<style>` sollte nicht leer sein
- âŒ Wenn `Detected Theme: unknown` â†’ Theme nicht erkannt!
- âŒ Wenn `Loaded CSS files:` leer â†’ Keine passende CSS-Datei gefunden!

**HÃ¤ufige Probleme:**
```html
<!-- Detected Theme: _jc-theme-justlight_ -->
<!-- Loaded CSS files: menu-icons_universal.css -->
```
â†’ âŒ Es gibt keine `menu-icons_justlight_.css` â†’ Fallback auf universal.css

**LÃ¶sung:** CSS-Datei fÃ¼r Theme erstellen (siehe Abschnitt 11)

---

**Methode 2: Browser-Konsole (fÃ¼r Entwickler)**

```javascript
// Browser-Konsole (F12)

// 1. Theme-Klasse prÃ¼fen
console.log(document.body.className);
// Erwartet: "wt-theme-webtrees"

// 2. CSS-Regel prÃ¼fen
let el = document.querySelector('.menu-list-burial');
let styles = window.getComputedStyle(el, '::before');
console.log(styles.backgroundImage);
// Erwartet: url("...")

// 3. Icon-URL testen
let img = new Image();
img.src = '/modules_v4/burial-places-report/resources/icons/menu-list-burial-webtrees.svg';
img.onload = () => console.log('âœ… Icon gefunden');
img.onerror = () => console.log('âŒ Icon NICHT gefunden');
```

**LÃ¶sungen:**
1. **CSS-Datei prÃ¼fen:**
   - Existiert `menu-icons.webtrees.css`?
   - Stimmt Theme-Name im Dateinamen?
   - Stimmt Theme-Klasse im CSS-Selektor?

2. **Icon-Datei prÃ¼fen:**
   - Existiert Icon-Datei?
   - Stimmt Pfad in CSS (`background-image`)?
   - Sind Dateiberechtigungen korrekt? (644)

3. **Browser-Cache leeren:**
   - Strg+F5 (Windows)
   - Cmd+Shift+R (Mac)

4. **Modul neu aktivieren:**
   - Verwaltung â†’ Module â†’ Burial Places deaktivieren
   - Seite neu laden
   - Burial Places wieder aktivieren

---

#### Problem 2: Icon zu groÃŸ/klein/falsch positioniert

**Symptome:**
- Icon ragt Ã¼ber MenÃ¼zeile hinaus
- Icon Ã¼berschneidet Text
- Icon zu weit oben/unten/links/rechts

**LÃ¶sung:**

CSS-Datei Ã¶ffnen und Parameter anpassen:

```css
/* Icon-GrÃ¶ÃŸe */
width: 1.50em;   /* Ã„ndern: 1.00em - 2.00em */
height: 1.50em;

/* Horizontal */
left: 0.35em;    /* Ã„ndern: 0.20em - 0.80em */

/* Vertikal */
top: 32%;        /* Ã„ndern: 20% - 50% */

/* Platz fÃ¼r Icon */
padding-left: 2.20em !important;  /* Ã„ndern: 2.00em - 3.00em */
```

Siehe [Abschnitt 13: CSS-Parameter](#13-css-parameter-im-detail) fÃ¼r Details.

---

#### Problem 3: "An error occurred"

**Symptome:**
- Fehlermeldung beim Ã–ffnen der Liste
- WeiÃŸe Seite (White Screen of Death)

**Diagnose:**
```bash
# Webtrees-Log prÃ¼fen
tail -f data/logs/webtrees.log

# PHP-Error-Log prÃ¼fen
tail -f /var/log/php-error.log

# Apache/Nginx Error-Log prÃ¼fen
tail -f /var/log/apache2/error.log
```

**HÃ¤ufige Ursachen:**

1. **PHP-Version zu alt:**
   ```
   PHP Fatal error: ... requires PHP 8.0 or higher
   ```
   â†’ PHP auf 8.0+ aktualisieren

2. **Fehlende PHP-Erweiterungen:**
   ```
   PHP Fatal error: Call to undefined function mb_strlen()
   ```
   â†’ PHP-Erweiterung `mbstring` installieren

3. **Speicher-Limit:**
   ```
   PHP Fatal error: Allowed memory size of ... exhausted
   ```
   â†’ `memory_limit` in `php.ini` erhÃ¶hen (z.B. auf 256M)

4. **Syntax-Fehler in module.php:**
   ```
   PHP Parse error: syntax error, unexpected ...
   ```
   â†’ Datei von Originalversion wiederherstellen

---

#### Problem 4: CSV-Export enthÃ¤lt keine Daten

**Symptome:**
- CSV-Datei ist leer (nur Header)
- CSV-Datei wird nicht heruntergeladen

**Diagnose:**

1. **Browser-Konsole prÃ¼fen (F12):**
   ```
   Network-Tab â†’ POST request â†’ Response prÃ¼fen
   ```

2. **Popup-Blocker prÃ¼fen:**
   - Oft blockiert Popup-Blocker den Download

**LÃ¶sungen:**

1. **Keine Daten gefunden:**
   - Filter zu restriktiv â†’ Filter lockern
   - Keine Bestattungen im Baum â†’ Testdaten hinzufÃ¼gen

2. **Server-Timeout:**
   - Zu viele Daten â†’ Filter verwenden
   - `max_execution_time` in `php.ini` erhÃ¶hen

3. **Falsche HTTP-Header:**
   - PrÃ¼fe `exportCsvAction()` Methode
   - `Content-Type` muss `text/csv` sein
   - `Content-Disposition` muss `attachment` sein

---

#### Problem 5: Keyword-Validierung schlÃ¤gt fehl

**Symptome:**
- "Too many keywords" Fehler
- "Keyword too long" Fehler

**LÃ¶sung:**

1. **Zu viele Keywords (>20):**
   ```
   Friedhof,Cemetery,CimetiÃ¨re,...
   ```
   â†’ Keywords reduzieren auf maximal 20

2. **Keyword zu lang (>50 Zeichen):**
   ```
   Friedhof-der-Namenlosen-und-Vergessenen-Menschen
   ```
   â†’ Keyword kÃ¼rzen oder aufteilen

3. **GesamtlÃ¤nge zu groÃŸ (>250 Zeichen):**
   ```
   Friedhof,Cemetery,CimetiÃ¨re,... (insgesamt 280 Zeichen)
   ```
   â†’ Weniger oder kÃ¼rzere Keywords verwenden

---

### 23.2 Debug-Modus aktivieren

**In `module.php` hinzufÃ¼gen:**

```php
// Am Anfang der Klasse
private const DEBUG = true;

// In Methoden
if (self::DEBUG) {
    error_log('Burial Places Debug: ' . print_r($data, true));
}
```

**Oder in `index.php`:**

```php
// Vor require 'includes/session.php';
define('WT_DEBUG', true);
error_reporting(E_ALL);
ini_set('display_errors', '1');
```

### 23.3 CSS-Debugging

**CSS-Regeln inspizieren:**

1. **DevTools Ã¶ffnen (F12)**
2. **Element auswÃ¤hlen** (MenÃ¼punkt "Bestattungsorte")
3. **Styles-Tab Ã¶ffnen**
4. **PrÃ¼fe welche Regeln angewendet werden:**
   ```
   .wt-theme-webtrees .dropdown-item.menu-list-burial
     line-height: 1.90 !important;  â† Aktive Regel
     padding: ...
   ```
5. **PrÃ¼fe ::before Pseudo-Element:**
   ```
   ::before
     background-image: url(...)  â† Icon-URL
     width: 1.50em
     height: 1.50em
   ```

---

# Teil 6: Erweiterung und Wartung

## 24. Erweiterbarkeit

### 24.1 Neue Features hinzufÃ¼gen

Das Modul ist so strukturiert, dass neue Features leicht hinzugefÃ¼gt werden kÃ¶nnen:

#### Feature: Dashboard-Widget

```php
// 1. Interface implementieren
use Fisharebest\Webtrees\Module\ModuleDashboardInterface;
use Fisharebest\Webtrees\Module\ModuleDashboardTrait;

class BurialPlacesReportModule extends AbstractModule implements 
    ModuleCustomInterface,
    ModuleListInterface,
    ModuleConfigInterface,
    ModuleGlobalInterface,
    ModuleDashboardInterface  // Neu
{
    use ModuleDashboardTrait;  // Neu
    
    // 2. Widget-Methode implementieren
    public function dashboardWidget(Tree $tree): string
    {
        // Statistiken sammeln
        $total_burials = 0;
        $with_cemetery = 0;
        
        foreach ($tree->individuals() as $individual) {
            $burials = $individual->facts(['BURI']);
            foreach ($burials as $burial) {
                $total_burials++;
                
                $result = $this->determineBurialPlace($burial, $tree, ...);
                if ($result['source'] !== 'NONE') {
                    $with_cemetery++;
                }
            }
        }
        
        // Widget rendern
        return view($this->name() . '::dashboard-widget', [
            'total_burials' => $total_burials,
            'with_cemetery' => $with_cemetery,
            'percentage' => round($with_cemetery / $total_burials * 100, 1),
        ]);
    }
}
```

#### Feature: Karten-Ansicht

```php
// 1. Neue Route registrieren
public function boot(): void
{
    // ... bestehende Routen ...
    
    Registry::routeFactory()->routeMap()
        ->get('burial-places-map', '/tree/{tree}/burial-places/map', [$this, 'mapAction']);
}

// 2. Controller-Methode
public function mapAction(ServerRequestInterface $request): ResponseInterface
{
    $tree = Validator::attributes($request)->tree();
    
    // Daten mit Koordinaten sammeln
    $institutions = $this->collectInstitutionsWithCoordinates($tree);
    
    // Karten-View rendern
    return $this->viewResponse($this->name() . '::map-page', [
        'title' => I18N::translate('Burial Places Map'),
        'tree' => $tree,
        'institutions' => $institutions,
    ]);
}

// 3. Koordinaten ermitteln
private function collectInstitutionsWithCoordinates(Tree $tree): array
{
    $institutions = $this->collectInstitutions($tree, ...);
    
    foreach ($institutions as &$data) {
        // Geocoding (z.B. via OpenStreetMap Nominatim API)
        $coords = $this->geocode($data['place']);
        $data['latitude'] = $coords['lat'];
        $data['longitude'] = $coords['lon'];
    }
    
    return $institutions;
}

// 4. View erstellen: resources/views/map-page.phtml
```

---

#### Feature: JSON-API

```php
// 1. Route registrieren
Registry::routeFactory()->routeMap()
    ->get('burial-places-api', '/api/burial-places/{tree}', [$this, 'apiAction']);

// 2. API-Controller
public function apiAction(ServerRequestInterface $request): ResponseInterface
{
    $tree = Validator::attributes($request)->tree();
    
    // API-Key prÃ¼fen (optional)
    $api_key = $request->getHeaderLine('X-API-Key');
    if ($api_key !== $this->getModulePreference('api_key')) {
        return response('Unauthorized', 401);
    }
    
    // Daten sammeln
    $institutions = $this->collectInstitutions($tree, ...);
    
    // In API-Format konvertieren
    $api_data = array_map(function($data) {
        return [
            'place' => $data['place'],
            'institution' => $data['institution'],
            'count' => count($data['individuals']),
            'individuals' => array_map(function($entry) {
                return [
                    'xref' => $entry['individual']->xref(),
                    'name' => $entry['individual']->fullName(),
                    'burial_date' => $entry['burial']->date()->format('Y-m-d'),
                ];
            }, $data['individuals']),
        ];
    }, $institutions);
    
    // JSON-Response
    return response(json_encode($api_data, JSON_PRETTY_PRINT))
        ->withHeader('Content-Type', 'application/json');
}
```

---

### 24.2 Neue Themes unterstÃ¼tzen

**Schritt 1:** CSS-Datei erstellen
```
menu-icons.neues-theme.css
```

**Schritt 2:** Icon-Datei hinzufÃ¼gen
```
menu-list-burial-neues-theme.svg
```

**Schritt 3:** Fertig! (headContent() lÃ¤dt automatisch)

---

### 24.3 Neue Sprachen hinzufÃ¼gen

**Schritt 1:** Sprachdatei erstellen
```php
// resources/lang/es.php (Spanisch)
<?php

return [
    'Burial Places Report' => 'Informe de Lugares de Sepultura',
    'Burial place' => 'Lugar de sepultura',
    'Cemetery/Institution' => 'Cementerio/InstituciÃ³n',
    // ... weitere Ãœbersetzungen
];
```

**Schritt 2:** Fertig! (Webtrees lÃ¤dt automatisch)

---

## 25. Wartung und Updates

### 25.1 Update-Prozess

**Vorbereitung:**
1. Backup erstellen (Dateien + Datenbank)
2. Webtrees in Wartungsmodus versetzen
3. Update-Notizen lesen

**Update durchfÃ¼hren:**
```bash
# 1. Altes Modul sichern
mv burial-places-report burial-places-report.backup

# 2. Neues Modul hochladen
unzip burial-places-report-v2.2.4.1.1.zip
mv burial-places-report modules_v4/

# 3. Dateiberechtigungen setzen
chmod -R 644 modules_v4/burial-places-report/*
chmod 755 modules_v4/burial-places-report
chmod 755 modules_v4/burial-places-report/resources
chmod 755 modules_v4/burial-places-report/resources/*

# 4. Webtrees neu laden
# Browser: Strg+F5

# 5. Modul-Einstellungen prÃ¼fen
# Admin â†’ Module â†’ Listen â†’ Bestattungsorte (Zahnrad)
```

**Nach Update:**
1. Alle Einstellungen prÃ¼fen
2. Testliste erstellen
3. CSV-Export testen
4. Verschiedene Themes testen

---

### 25.2 Datenbank-Migrationen

**Falls Datenbank-Ã„nderungen nÃ¶tig:**

```php
// In boot() Methode
public function boot(): void
{
    // Versions-Check
    $current_version = $this->getModulePreference('version', '0.0.0');
    
    if (version_compare($current_version, '2.2.4.1.1', '<')) {
        $this->migrate();
    }
    
    // ... Rest der boot() Methode
}

// Migrations-Methode
private function migrate(): void
{
    // Migration 1: Neue Preferences hinzufÃ¼gen
    if ($this->getModulePreference('version', '0.0.0') === '0.0.0') {
        // Alte Daten migrieren
        foreach (Tree::all() as $tree) {
            $old_keywords = $tree->getPreference('burial-places-report-plac-keywords');
            if ($old_keywords) {
                $this->setTreePreference($tree, 'plac_keywords', $old_keywords);
            }
        }
    }
    
    // Version aktualisieren
    $this->setModulePreference('version', self::CUSTOM_VERSION);
}
```

---

### 25.3 KompatibilitÃ¤t prÃ¼fen

**Webtrees-Version:**
```php
use Fisharebest\Webtrees\Webtrees;

if (version_compare(Webtrees::VERSION, '2.2.1', '<')) {
    throw new \RuntimeException('Requires Webtrees 2.2.1 or higher');
}
```

**PHP-Version:**
```php
if (version_compare(PHP_VERSION, '8.0.0', '<')) {
    throw new \RuntimeException('Requires PHP 8.0 or higher');
}
```

**PHP-Erweiterungen:**
```php
$required_extensions = ['mbstring', 'pdo', 'gd'];

foreach ($required_extensions as $ext) {
    if (!extension_loaded($ext)) {
        throw new \RuntimeException("Missing PHP extension: $ext");
    }
}
```

---

## 26. Best Practices

### 26.1 Code-Style

**PSR-12 Standard befolgen:**
```php
// âœ… Richtig
class BurialPlacesReportModule extends AbstractModule
{
    private const DEFAULT_VALUE = 'default';
    
    public function methodName(string $param): string
    {
        if ($condition) {
            return $result;
        }
        
        return '';
    }
}

// âŒ Falsch
class BurialPlacesReportModule extends AbstractModule {
  private const DEFAULT_VALUE="default";
  public function methodName($param):string{
    if($condition)return $result;
    return '';
  }
}
```

**Namenskonventionen:**
```php
// Klassen: PascalCase
class BurialPlacesReportModule

// Methoden: camelCase
public function getConfigLink()

// Variablen: snake_case
$fallback_order
$plac_keywords

// Konstanten: UPPER_SNAKE_CASE
private const DEFAULT_FALLBACK_ORDER
```

---

### 26.2 Dokumentation

**PHPDoc-Kommentare:**
```php
/**
 * Ermittelt Friedhof/Institution aus GEDCOM-Daten
 * 
 * Diese Methode durchsucht GEDCOM-Felder in konfigurierter Reihenfolge
 * und verwendet Keyword-Matching fÃ¼r PLAC und NOTE Felder.
 * 
 * @param Fact  $burial        BURI-Fact Objekt
 * @param Tree  $tree          Stammbaum-Objekt
 * @param array $sources       Reihenfolge: ['AGNC', 'PLAC', 'NOTE']
 * @param array $keywords      Keyword-Liste fÃ¼r PLAC/NOTE
 * 
 * @return array {
 *     @type string $value   Friedhofsname oder 'Unknown'
 *     @type string $source  Quelle: 'AGNC'|'NOTE'|'PLAC'|'NONE'
 * }
 * 
 * @throws \RuntimeException Wenn Tree nicht geladen werden kann
 */
private function determineBurialPlace(
    Fact $burial,
    Tree $tree,
    array $sources,
    array $keywords
): array {
    // ...
}
```

---

### 26.3 Testing

**Manuelle Tests:**
```
1. Fresh Install:
   - Modul installieren auf frischer Webtrees-Installation
   - Alle Features testen
   
2. Upgrade Test:
   - Alte Version installieren
   - Auf neue Version updaten
   - Einstellungen prÃ¼fen
   - Features testen
   
3. Theme-Tests:
   - Alle offiziellen Themes testen
   - Icons prÃ¼fen
   - Layout prÃ¼fen
   
4. Browser-Tests:
   - Chrome/Edge (Chromium)
   - Firefox
   - Safari
   - Mobile (Chrome/Safari)
   
5. Daten-Tests:
   - Leerer Baum (keine Bestattungen)
   - Kleiner Baum (< 100 Personen)
   - Mittlerer Baum (100-1000 Personen)
   - GroÃŸer Baum (> 1000 Personen)
   
6. Edge Cases:
   - Keine AGNC-Felder
   - Keine Keywords in PLAC/NOTE
   - Umlaute in Namen
   - Sonderzeichen in Notizen
   - Sehr lange Friedhofsnamen
```

---

## 27. Code-Beispiele

### 27.1 Minimal-Modul (ohne Icons)

```php
<?php

declare(strict_types=1);

namespace MyVendor\Webtrees\Module\BurialPlacesReport;

use Fisharebest\Webtrees\Module\AbstractModule;
use Fisharebest\Webtrees\Module\ModuleCustomInterface;
use Fisharebest\Webtrees\Module\ModuleCustomTrait;
use Fisharebest\Webtrees\Module\ModuleListInterface;
use Fisharebest\Webtrees\Module\ModuleListTrait;
use Fisharebest\Webtrees\View;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

class BurialPlacesReportModule extends AbstractModule implements 
    ModuleCustomInterface,
    ModuleListInterface
{
    use ModuleCustomTrait;
    use ModuleListTrait;

    public function boot(): void
    {
        View::registerNamespace($this->name(), $this->resourcesFolder() . 'views/');
    }

    public function title(): string
    {
        return 'Burial Places Report';
    }

    public function description(): string
    {
        return 'A list of burials, grouped by burial sites.';
    }

    public function listMenuClass(): string
    {
        return 'menu-list-burial';
    }

    public function listUrl(Tree $tree, array $parameters = []): string
    {
        return route('burial-places-list', ['tree' => $tree->name()]);
    }

    public function handle(ServerRequestInterface $request): ResponseInterface
    {
        // ... Daten sammeln und View rendern
    }
}
```

---

### 27.2 Icon-Integration hinzufÃ¼gen

```php
// Interface hinzufÃ¼gen
use Fisharebest\Webtrees\Module\ModuleGlobalInterface;
use Fisharebest\Webtrees\Module\ModuleGlobalTrait;

class BurialPlacesReportModule extends AbstractModule implements 
    ModuleCustomInterface,
    ModuleListInterface,
    ModuleGlobalInterface  // Neu
{
    use ModuleCustomTrait;
    use ModuleListTrait;
    use ModuleGlobalTrait;  // Neu
    
    // headContent() implementieren
    public function headContent(): string
    {
        // Siehe Anhang A fÃ¼r vollstÃ¤ndigen Code
    }
}
```

---

# AnhÃ¤nge

## Anhang A: VollstÃ¤ndiger Code headContent()

```php
/**
 * FÃ¼gt CSS fÃ¼r Icons in <head> ein
 * 
 * @return string CSS in <style>-Tags
 */
public function headContent(): string
{
    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    // 1. AGNC-FUNKTIONALITÃ„T
    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    $html = view($this->name() . '::agnc-always-show', [
        'module_name' => $this->name(),
    ]);

    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    // 2. PFADE VORBEREITEN
    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    
    // Dateisystem-Pfad zu CSS-Verzeichnis
    $css_dir_fs  = rtrim($this->resourcesFolder(), '/\\') . '/css/';
    
    // Modul-Verzeichnisname (z.B. "burial-places-report")
    $module_dir  = basename(dirname(__FILE__));
    
    // Basis-URL fÃ¼r Icons (wird in CSS-Dateien eingefÃ¼gt)
    $icons_base  = '/modules_v4/' . $module_dir . '/resources/icons/';

    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    // 3. AKTUELLES THEME ERMITTELN
    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    
    $current_theme = '';
    try {
        $request = Registry::container()->get(ServerRequestInterface::class);
        $theme_attr = $request->getAttribute('theme');
        
        if ($theme_attr !== null && method_exists($theme_attr, 'name')) {
            $current_theme = $theme_attr->name();
        }
    } catch (\Exception $e) {
        // Fallback: Theme konnte nicht ermittelt werden
        $current_theme = '';
    }

    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    // 4. CSS-DATEIEN SAMMELN
    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    
    $files = [];
    $theme_css_found = false;

    // Wenn Theme ermittelt wurde: nur dessen CSS laden
    if ($current_theme !== '') {
        $theme_css = 'menu-icons.' . $current_theme . '.css';
        if (is_file($css_dir_fs . $theme_css)) {
            $files[] = $theme_css;
            $theme_css_found = true;
        }
    }

    // FALLBACK 1: Wenn kein Theme-CSS existiert, universal.css
    if (!$theme_css_found && is_file($css_dir_fs . 'menu-icons.universal.css')) {
        $files[] = 'menu-icons.universal.css';
        $theme_css_found = true;
    }

    // FALLBACK 2: Wenn gar nichts gefunden, lade alle Theme-CSS
    if (!$theme_css_found) {
        $all = glob($css_dir_fs . 'menu-icons.*.css');
        if (is_array($all)) {
            foreach ($all as $abs) {
                $bn = basename($abs);
                
                // AusschlieÃŸen: universal, none, template-*
                if ($bn === 'menu-icons.universal.css' || 
                    $bn === 'menu-icons.none.css' || 
                    str_starts_with($bn, 'menu-icons.template-')) {
                    continue;
                }
                
                // Nur echte Theme-Dateien
                if (preg_match('/^menu-icons\..+\.css$/', $bn)) {
                    $files[] = $bn;
                }
            }
        }
    }

    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    // 5. CSS-INHALTE ZUSAMMENFASSEN
    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    
    $bundle = '';
    foreach ($files as $file) {
        $path = $css_dir_fs . $file;
        if (!is_file($path)) {
            continue;
        }

        $css = file_get_contents($path);
        if ($css === false) {
            continue;
        }

        // Pfadfix: relative â†’ absolute (fÃ¼r inline <style>)
        // a) ../icons/webtrees/... â†’ /modules_v4/<modul>/resources/icons/webtrees/...
        $css = str_replace('../icons/webtrees/', $icons_base . 'webtrees/', $css);
        // b) ../icons/... â†’ /modules_v4/<modul>/resources/icons/...
        $css = str_replace('../icons/', $icons_base, $css);
        // c) Windows-Pfade normalisieren
        $css = str_replace('\\', '/', $css);

        // CSS mit Kommentar hinzufÃ¼gen
        $bundle .= "\n/* {$file} */\n" . $css . "\n";
    }

    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    // 6. CSS ALS <STYLE>-TAG AUSGEBEN
    // â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
    
    if ($bundle !== '') {
        $html .= "\n<style>\n" . $bundle . "\n</style>\n";
    }

    return $html;
}
```

---

## Anhang B: Komplette CSS-Datei Beispiel

```css
/* ============================================================
   webtrees â€“ MenÃ¼punkt "Bestattungsorte"
   ============================================================
   DE: Diese Datei wirkt NUR, wenn das webtrees-Theme aktiv ist.
   EN: This file only applies when the webtrees theme is active.
   ============================================================ */

/* â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘
   ZEILENLAYOUT / ROW LAYOUT
   â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘ */

.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial {
  /* DE: Positionierung - NICHT Ã¤ndern!
     EN: Positioning - DO NOT change! */
  position: relative;
  
  /* DE: ZeilenhÃ¶he des MenÃ¼punkts (hÃ¶here Werte = mehr Abstand zwischen Zeilen)
     EN: Line height of menu item (higher values = more space between lines)
     Werte / Values: 1.0 - 3.0 (empfohlen / recommended: 1.5 - 2.0) */
  line-height: 1.90 !important;
  
  /* DE: InnenabstÃ¤nde des MenÃ¼punkts (oben rechts unten links)
     EN: Inner padding of menu item (top right bottom left)
     Format: padding: [oben/top] [rechts/right] [unten/bottom] [links/left]
     Empfohlen / Recommended: padding-left sollte 2.0em - 2.5em sein fÃ¼r Icon */
  padding: 0rem 1rem 0.70rem 2.20em !important;
}


/* â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘
   ICON VOR DEM TEXT / ICON BEFORE TEXT
   â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘â–‘ */

.wt-theme-webtrees .wt-genealogy-menu .dropdown-item.menu-list-burial::before {
  /* DE: Leeres Pseudo-Element fÃ¼r das Icon - NICHT Ã¤ndern!
     EN: Empty pseudo-element for icon - DO NOT change! */
  content: "";
  
  /* DE: Positionierung - NICHT Ã¤ndern!
     EN: Positioning - DO NOT change! */
  position: absolute;
  
  /* DE: Horizontale Position des Icons (vom linken Rand)
     EN: Horizontal position of icon (from left edge)
     Werte / Values: 0.2em - 0.8em
     Kleiner = weiter links / Smaller = further left
     GrÃ¶ÃŸer = weiter rechts / Larger = further right */
  left: 0.35em;
  
  /* DE: Vertikale Position des Icons (von oben)
     EN: Vertical position of icon (from top)
     Werte / Values: 20% - 50%
     Kleiner = hÃ¶her / Smaller = higher
     GrÃ¶ÃŸer = tiefer / Larger = lower */
  top: 32%;
  
  /* DE: Zentrierung - NICHT Ã¤ndern!
     EN: Centering - DO NOT change! */
  transform: translateY(-50%);
  
  /* DE: Breite des Icons
     EN: Width of icon
     Werte / Values: 0.8em - 1.5em (empfohlen / recommended: 1.0em - 1.2em) */
  width: 1.50em;
  
  /* DE: HÃ¶he des Icons
     EN: Height of icon
     Werte / Values: 0.8em - 1.5em (empfohlen / recommended: 1.0em - 1.2em) */
  height: 1.50em;
  
  /* DE: Pfad zum Icon (relativ zur CSS-Datei)
     EN: Path to icon (relative to CSS file)
     Anpassen / Customize: Ersetze den Dateinamen / Replace the filename
     Beispiel / Example: "../icons/mein-icon.svg" */
  background-image: url("../icons/menu-list-burial-webtrees.svg");
  
  /* DE: Icon-Wiederholung - NICHT Ã¤ndern!
     EN: Icon repetition - DO NOT change! */
  background-repeat: no-repeat;
  
  /* DE: Icon-Skalierung - NICHT Ã¤ndern fÃ¼r beste QualitÃ¤t!
     EN: Icon scaling - DO NOT change for best quality! */
  background-size: contain;
  
  /* DE: Icon-Ausrichtung - NICHT Ã¤ndern!
     EN: Icon alignment - DO NOT change! */
  background-position: center;
  
  /* DE: Verhindert Klick-Probleme - NICHT Ã¤ndern!
     EN: Prevents click issues - DO NOT change! */
  pointer-events: none;
}


/* â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
   SCHNELLANLEITUNG / QUICK GUIDE
   â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•
   
   DE: Was kann ich anpassen?
   1. Icon grÃ¶ÃŸer/kleiner machen: width und height Ã¤ndern (z.B. 1.2em / 0.9em)
   2. Icon horizontal verschieben: left Ã¤ndern (z.B. 0.5em = weiter rechts)
   3. Icon vertikal verschieben: top Ã¤ndern (z.B. 40% = tiefer)
   4. Text mehr Platz links: padding-left erhÃ¶hen (z.B. 2.5em)
   5. ZeilenhÃ¶he anpassen: line-height Ã¤ndern (z.B. 2.0)
   6. Vertikale AbstÃ¤nde: padding-top und padding-bottom anpassen
   7. Anderes Icon verwenden: background-image Pfad Ã¤ndern
   
   EN: What can I customize?
   1. Make icon larger/smaller: change width and height (e.g. 1.2em / 0.9em)
   2. Move icon horizontally: change left (e.g. 0.5em = further right)
   3. Move icon vertically: change top (e.g. 40% = lower)
   4. More text space on left: increase padding-left (e.g. 2.5em)
   5. Adjust line height: change line-height (e.g. 2.0)
   6. Vertical spacing: adjust padding-top and padding-bottom
   7. Use different icon: change background-image path
   
   â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â•â• */
```

---

## Anhang C: Performance-Benchmarks

**Test-Umgebung:**
- **Server:** Apache 2.4, PHP 8.3, MySQL 8.0
- **Hardware:** 4 GB RAM, 2 CPU Cores
- **Webtrees:** Version 2.2.4
- **Modul:** Version 2.2.4.1.1

**Test-Daten:**
- 5.000 Personen
- 3.500 Bestattungen
- 850 verschiedene FriedhÃ¶fe
- 120 verschiedene Bestattungsorte

**Ergebnisse:**

| Aktion | Zeit | Speicher | Notizen |
|--------|------|----------|---------|
| **Report-Generierung** (ohne Filter) | 2.5s | 80 MB | Alle Bestattungen |
| **Report-Generierung** (mit Filter) | 1.2s | 45 MB | ~500 Bestattungen |
| **CSV-Export** | 3.0s | 85 MB | Alle Bestattungen |
| **Admin-Seite laden** | 0.1s | 15 MB | Einstellungen |
| **headContent() AusfÃ¼hrung** | 0.02s | 2 MB | CSS laden |
| **Erste Seite (inkl. Cache)** | 2.8s | 82 MB | Cold cache |
| **Zweite Seite (Cache warm)** | 0.8s | 45 MB | Warm cache |

**Skalierung:**

| Personen | Bestattungen | Zeit | Speicher |
|----------|--------------|------|----------|
| 1.000 | 700 | 0.5s | 25 MB |
| 5.000 | 3.500 | 2.5s | 80 MB |
| 10.000 | 7.000 | 5.2s | 155 MB |
| 25.000 | 17.500 | 13.5s | 380 MB |

**Optimierungen:**
- Bei >10.000 Personen: Caching implementieren
- Bei >25.000 Personen: Pagination implementieren
- Filter verwenden fÃ¼r schnellere Ergebnisse

---

## Anhang D: KompatibilitÃ¤ts-Matrix

### Webtrees-Versionen

| Webtrees | PHP | Modul | Getestet | Status | Notizen |
|----------|-----|-------|----------|--------|---------|
| 2.2.1 | 8.0 | 2.2.4.1.1 | Ja | âœ… | Stabil |
| 2.2.2 | 8.1 | 2.2.4.1.1 | Ja | âœ… | Stabil |
| 2.2.3 | 8.2 | 2.2.4.1.1 | Ja | âœ… | Stabil |
| 2.2.4 | 8.3 | 2.2.4.1.1 | Ja | âœ… | Stabil |
| 2.1.x | 8.0 | 2.2.4.1.1 | Nein | â“ | Ungetestet |
| 2.3.x | 8.3 | 2.2.4.1.1 | Nein | â“ | Nicht verfÃ¼gbar |

### PHP-Versionen

| PHP | Status | Notizen |
|-----|--------|---------|
| 7.4 | âŒ | Zu alt, nicht unterstÃ¼tzt |
| 8.0 | âœ… | Minimum, funktioniert |
| 8.1 | âœ… | Empfohlen |
| 8.2 | âœ… | Empfohlen |
| 8.3 | âœ… | Empfohlen, beste Performance |

### Browser-KompatibilitÃ¤t

| Browser | Desktop | Mobile | Status | Notizen |
|---------|---------|--------|--------|---------|
| **Chrome** | 90+ | 90+ | âœ… | VollstÃ¤ndig getestet |
| **Edge** (Chromium) | 90+ | - | âœ… | VollstÃ¤ndig getestet |
| **Firefox** | 88+ | 88+ | âœ… | VollstÃ¤ndig getestet |
| **Safari** | 14+ | 14+ | âœ… | Getestet |
| **Opera** | 76+ | - | âœ… | Funktioniert (Chromium) |
| **IE11** | - | - | âŒ | Nicht unterstÃ¼tzt |

### Theme-KompatibilitÃ¤t

| Theme | CSS-Datei | Icon | Status |
|-------|-----------|------|--------|
| **webtrees** | âœ… | âœ… | VollstÃ¤ndig unterstÃ¼tzt |
| **colors** | âœ… | âœ… | VollstÃ¤ndig unterstÃ¼tzt |
| **fab** | âœ… | âœ… | VollstÃ¤ndig unterstÃ¼tzt |
| **minimal** | âœ… | âŒ | UnterstÃ¼tzt (ohne Icon) |
| **clouds** | âœ… | âœ… | VollstÃ¤ndig unterstÃ¼tzt |
| **xenea** | âœ… | âœ… | VollstÃ¤ndig unterstÃ¼tzt |
| **Andere** | âœ… (universal) | âœ… | Fallback funktioniert |

### Genealogie-Programme

| Programm | AGNC | PLAC | NOTE | Status |
|----------|------|------|------|--------|
| **Ahnenblatt** | âœ… | âœ… | âœ… | Getestet |
| **Gramps** | âš ï¸ | âœ… | âœ… | AGNC selten, NOTE bevorzugt |
| **Family Tree Maker** | âœ… | âœ… | âš ï¸ | PLAC bevorzugt |
| **Legacy** | âœ… | âœ… | âœ… | Getestet |
| **Ancestris** | âœ… | âœ… | âœ… | Funktioniert |
| **RootsMagic** | âœ… | âœ… | âœ… | Funktioniert |

---

## Changelog

### Version 2.2.4.1.2 (Aktuell - Oktober 2025)

**Bug-Fixes:**
- 🐛 **KRITISCH:** Universelle Pfad-Erkennung für Icon-Darstellung implementiert
  - **Problem:** Icons wurden nicht angezeigt bei Installationen in Unterverzeichnissen
  - **Beispiel:** Installation unter `https://example.com/wt-test2/` → Icons nicht sichtbar
  - **Ursache:** Hardcodierte Icon-Pfade ohne Base-Path in `headContent()` Methode
    ```php
    // ❌ ALT (funktioniert nur bei Root/Subdomain):
    $icons_base = '/modules_v4/' . $module_dir . '/resources/icons/';
    ```
  - **Lösung:** Dynamische Base-Path-Ermittlung über `$_SERVER['SCRIPT_NAME']`
    ```php
    // ✅ NEU (funktioniert überall):
    $scriptName = $_SERVER['SCRIPT_NAME'] ?? '/index.php';
    $basePath = dirname($scriptName);
    if ($basePath === '/' || $basePath === '\\' || $basePath === '.') {
        $basePath = '';
    }
    $icons_base = $basePath . '/modules_v4/' . $module_dir . '/resources/icons/';
    ```
  - **Betroffen:** `module.php` - `headContent()` Methode (Zeilen 149-159)
  - **Ergebnis:**
    - ✅ Root-Installation: `https://example.com/` → `/modules_v4/.../icons/...`
    - ✅ Subdomain: `https://stammbaum.example.com/` → `/modules_v4/.../icons/...`
    - ✅ Unterverzeichnis: `https://example.com/wt-test2/` → `/wt-test2/modules_v4/.../icons/...`

**Technische Details:**
- 🔧 Base-Path wird dynamisch aus Server-Variable ermittelt
- 🔧 Normalisierung für Root-Pfad (`/`, `\`, `.` → leer)
- 🔧 Kompatibel mit allen Webtrees-Installationsszenarien
- ✅ Alle anderen Funktionen bleiben unverändert
- ✅ Keine Auswirkungen auf bestehende Konfigurationen

**Dokumentation:**
- 📝 Basiert auf "Supplement: Universelle Pfad-Erkennung für Webtrees-Module"
- 📝 Referenz: Kapitel 7.5 im Webtrees Modul-Entwicklungshandbuch

---

### Version 2.2.4.1.1 (Oktober 2025)

**Bug-Fixes:**
- 🐛 **KRITISCH:** "Institution"-Label-Ersetzung durch "Friedhof" erfolgt jetzt nur noch bei BURI-Ereignissen
  - Problem: JavaScript hat "Institution" bei ALLEN Ereignissen ersetzt (z.B. auch bei "Grundausbildung")
  - Lösung: Neue `isBuriEvent()` Funktion prüft Ereignis-Kontext vor jeder Label-Ersetzung
  - Betroffen: `agnc-always-show.phtml` (Zeilen 57-116)
  - Prüfmethoden: GEDCOM-Code, Data-Attribute, CSS-Klassen, Kontext-Analyse, Formular-Validierung

**Technische Details:**
- 🔧 Neue Kontext-Prüfung in `changeInstitutionLabels()` Funktion
- 🔧 Verbesserte DOM-Analyse für präzise Ereignis-Erkennung
- ✅ Alle anderen Funktionen bleiben unverändert

---

### Version 2.2.4.1.0 (Oktober 2025)

**Neue Features:**
- ✨ Theme-spezifische Icon-Unterstützung
- ✨ Universal-CSS als Fallback
- ✨ Automatische CSS-Pfad-Korrektur
- ✨ Erweiterte Dokumentation

**Verbesserungen:**
- 🔧 `headContent()` Methode optimiert
- 🔧 CSS-Loading-Mechanismus verbessert
- 🔧 Icon-Pfad-Handling robuster
- 📝 Vollständige technische Dokumentation

**Bug-Fixes:**
- 🐛 Icon-Pfade auf Windows-Servern korrigiert
- 🐛 CSS-Spezifität-Probleme behoben

---

### Version 2.2.4.0.0 (September 2025)

**Neue Features:**
- ✨ Intelligente Fallback-Logik (AGNC → PLAC → NOTE)
- ✨ Konfigurierbare Reihenfolge (Drag & Drop)
- ✨ Keyword-Filter für PLAC/NOTE
- ✨ Diagnosemodus (zeigt Datenquelle)
- ✨ AGNC-Feld immer anzeigen Option
- ✨ Dropdown-Filter (mit/ohne Friedhofsinformation)
- ✨ CSV-Export mit vollständigen Daten
- ✨ Baumspezifische Konfiguration
- ✨ Reset-Funktion für Standardwerte
- ✨ Mehrsprachig (DE, EN, FR, HU)

**Verbesserungen:**
- 🔧 `validateKeywords()` Methode
- 🔧 `getTreePreference()` / `setTreePreference()` Hilfsmethoden
- 🔧 Verbesserte Validierung
- 🔧 Responsive Design
- 📝 Code-Dokumentation

**Bug-Fixes:**
- 🐛 Keyword-Deduplizierung
- 🐛 Preference-Namen vereinheitlicht
- 🐛 Filter-Logik korrigiert

---

**Ende der technischen Dokumentation**

*Version: 2.2.4.1.2*  
*Erstellt: Oktober 2025*  
*Autor: Thomas Schiller (mit Hilfe von Claude Sonnet 4.5)*  
*Lizenz: GPL v3.0*
